<?php
/**
 * KSeF Reader Header Rule class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Service\Reader\RecordBuilder\Rules;

use App\Integrations\KSeF\Service\Reader\RecordBuilder\AbstractRecordBuilder;

/**
 * KSeF Reader Header Rule class.
 *
 * Handles building header fields: P_2 (invoice number), P_1 (issue date),
 * P_6 (sale date), KSeF number, KSeF date, and invoice type.
 */
final class HeaderRule implements RuleInterface
{
	private AbstractRecordBuilder $builder;

	/** {@inheritDoc} */
	public function __construct(AbstractRecordBuilder $builder)
	{
		$this->builder = $builder;
	}

	/** {@inheritDoc} */
	public function apply(
		\Vtiger_Record_Model $record,
		array $context = [],
	): void {
		$this->builder->setFieldIfMapped($record, '//Fa/P_2', $this->builder->getRequiredString('//Fa/P_2'));

		$this->builder->setFieldIfMapped($record, '//Fa/P_1', $this->builder->getRequiredDate('//Fa/P_1'));

		if (!empty($context['xmlFileName'])) {
			$ksefNumber = pathinfo($context['xmlFileName'], PATHINFO_FILENAME);
			if ($ksefNumber) {
				$this->builder->setFieldIfMapped($record, 'yf_ksef_number', $ksefNumber);
			}
		}

		$fieldName = $this->builder->getMapperService()->getMapper()->get('yf_ksef_date')?->getField();
		if ($fieldName) {
			$record->set($fieldName, date('Y-m-d H:i:s'));
		}

		$p6 = $this->builder->getOptionalDate('//Fa/P_6');
		if ($p6) {
			$this->builder->setFieldIfMapped($record, '//Fa/P_6', $p6);
		}

		$invoiceType = $this->builder->getOptionalString('//Fa/RodzajFaktury');
		if ($invoiceType) {
			$this->builder->setFieldIfMapped($record, '//Fa/RodzajFaktury', $this->mapInvoiceType($invoiceType));
		}
	}

	/**
	 * Maps KSeF invoice type codes to picklist values.
	 *
	 * @param string $invoiceType The invoice type code from KSeF XML.
	 *
	 * @return string The corresponding picklist value.
	 */
	private function mapInvoiceType(string $invoiceType): string
	{
		return match ($invoiceType) {
			'VAT' => 'PLL_VAT',
			'ROZ' => 'PLL_ROZ',
			'ZAL' => 'PLL_ZAL',
			'UPR' => 'PLL_UPR',
			'KOR' => 'PLL_KOR',
			'KOR_ZAL' => 'PLL_KOR_ZAL',
			'KOR_ROZ' => 'PLL_KOR_ROZ',
			default => 'PLL_VAT',
		};
	}
}
