<?php

/**
 * KSeF Reader Payment Rule class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Service\Reader\RecordBuilder\Rules;

use App\Integrations\KSeF\Service\Reader\RecordBuilder\AbstractRecordBuilder;
use App\QueryGenerator;
use App\User;

/**
 * KSeF Reader Payment Rule class.
 */
final class PaymentRule implements RuleInterface
{
	private AbstractRecordBuilder $builder;

	/** {@inheritDoc} */
	public function __construct(AbstractRecordBuilder $builder)
	{
		$this->builder = $builder;
	}

	/** {@inheritDoc} */
	public function apply(
		\Vtiger_Record_Model $record,
		array $context = []
	): void {
		$this->applyPaymentDueDate($record);
		$this->applyPaymentMethod($record);
		$this->applyBankAccount($record);
		$this->applyPaymentUrl($record);
		$this->applyPaymentStatus($record);
		// $this->createPaymentRecords($record);
	}

	/*
	 * Create PaymentsOut record after invoice is saved.
	 *
	 * This method should be called after the invoice record is saved and has an ID.
	 */
	public function createPaymentRecords(\Vtiger_Record_Model $invoiceRecord): void
	{
		$isPaid = $this->builder->getOptionalString('//Fa/Platnosc/Zaplacono');
		if ('1' == $isPaid) {
			$this->createFullPaymentRecord($invoiceRecord);
		}

		$hasPartialPayments = $this->builder->getOptionalString('//Fa/Platnosc/ZnacznikZaplatyCzesciowej');
		if ('1' == $hasPartialPayments) {
			$this->createPartialPaymentRecords($invoiceRecord);
		}
	}

	public function applyPaymentStatus(\Vtiger_Record_Model $invoiceRecord): void
	{
		$isPaid = $this->builder->getOptionalString('//Fa/Platnosc/Zaplacono');
		$hasPartialPayments = $this->builder->getOptionalString('//Fa/Platnosc/ZnacznikZaplatyCzesciowej');

		$paymentStatus = match (true) {
			'1' == $isPaid || '2' == $hasPartialPayments => 'PLL_FULLY_PAID',
			'1' == $hasPartialPayments => 'PLL_PARTIALLY_PAID',
			default => 'PLL_AWAITING_PAYMENT'
		};

		$invoiceRecord->set('finvoicecost_paymentstatus', $paymentStatus);
	}

	/*
	 * Apply payment due date to invoice.
	 */
	private function applyPaymentDueDate(\Vtiger_Record_Model $record): void
	{
		$paymentDueDate = $this->builder->getOptionalDate('//Fa/Platnosc/TerminPlatnosci/Termin');
		if ($paymentDueDate) {
			$this->builder->setFieldIfMapped($record, '//Fa/Platnosc/TerminPlatnosci/Termin', $paymentDueDate);
		}
	}

	/*
	 * Apply payment method to invoice.
	 */
	private function applyPaymentMethod(\Vtiger_Record_Model $record): void
	{
		$paymentForm = $this->builder->getOptionalString('//Fa/Platnosc/FormaPlatnosci');
		if ($paymentForm) {
			$mappedMethod = $this->mapPaymentMethod($paymentForm);
			if ($mappedMethod) {
				$this->builder->setFieldIfMapped($record, '//Fa/Platnosc/FormaPlatnosci', $mappedMethod);
			}
		}
	}

	/*
	 * Apply payment URL to invoice.
	 */
	private function applyPaymentUrl(\Vtiger_Record_Model $record): void
	{
		$paymentUrl = $this->builder->getOptionalString('//Fa/Platnosc/LinkDoPlatnosci');
		if ($paymentUrl) {
			$this->builder->setFieldIfMapped($record, '//Fa/Platnosc/LinkDoPlatnosci', $paymentUrl);
		}
	}

	/**
	 * Apply bank account to invoice.
	 *
	 * @param \Vtiger_Record_Model $record
	 */
	private function applyBankAccount(\Vtiger_Record_Model $record): void
	{
		$bankAccountNumber = $this->builder->getOptionalString('//Fa/Platnosc/RachunekBankowy/NrRB');
		$this->builder->setFieldIfMapped($record, '//Fa/Platnosc/RachunekBankowy/NrRB', (string) $bankAccountNumber);

		/* Not implemented yet: RachunekBankowyFaktora */
	}

	/*
	 * Create PaymentsOut record for full payment.
	 */
	private function createFullPaymentRecord(\Vtiger_Record_Model $invoiceRecord): void
	{
		$paymentDate = $this->builder->getOptionalDate('//Fa/Platnosc/DataZaplaty');
		if (!$paymentDate) {
			return;
		}

		$paymentRecord = \Vtiger_Record_Model::getCleanInstance('PaymentsOut');
		$paymentRecord->set('assigned_user_id', User::getActiveAdminId());
		$paymentRecord->set('finvoiceid', $invoiceRecord->getId());
		$paymentRecord->set('payment_date', $paymentDate);
		$paymentRecord->set('paymentsvalue', $invoiceRecord->get('sum_gross') ?: 0);
		$paymentRecord->set('paymentsout_status', 'PLL_PAID');
		$paymentRecord->set('paymentstitle', 'Payment for invoice ' . $invoiceRecord->get('subject'));

		$currencyId = $invoiceRecord->get('currency_id');
		if ($currencyId) {
			$paymentRecord->set('currency_id', $currencyId);
		}

		$paymentRecord->save();
	}

	/*
	 * Create PaymentsOut records for partial payments.
	 */
	private function createPartialPaymentRecords(\Vtiger_Record_Model $invoiceRecord): void
	{
		$partialPaymentNodes = $this->builder->getXml()->xpath('//Fa/Platnosc/ZaplataCzesciowa');
		if (empty($partialPaymentNodes)) {
			return;
		}

		foreach ($partialPaymentNodes as $paymentNode) {
			$amount = $this->builder->getOptionalDecimal('KwotaZaplatyCzesciowej', $paymentNode);
			$date = $this->builder->getOptionalDate('DataZaplatyCzesciowej', $paymentNode);

			if (!$amount || !$date) {
				continue;
			}

			$paymentRecord = \Vtiger_Record_Model::getCleanInstance('PaymentsOut');
			$paymentRecord->set('finvoiceid', $invoiceRecord->getId());
			$paymentRecord->set('payment_date', $date);
			$paymentRecord->set('paymentsvalue', $amount);
			$paymentRecord->set('paymentsout_status', 'PLL_PAID');
			$paymentRecord->set('paymentstitle', 'Partial payment for invoice ' . $invoiceRecord->get('subject'));

			$currencyId = $invoiceRecord->get('currency_id');
			if ($currencyId) {
				$paymentRecord->set('currency_id', $currencyId);
			}

			$paymentRecord->save();
		}
	}

	private function mapPaymentMethod(string $paymentForm): ?string
	{
		return match ($paymentForm) {
			'1' => 'PLL_CASH',
			'2' => 'PLL_CARD',
			'3' => 'PLL_VOUCHER',
			'4' => 'PLL_CHECK',
			'5' => 'PLL_LOAN',
			'6' => 'PLL_TRANSFER',
			'7' => 'PLL_MOBILE',
			default => null,
		};
	}

	/*
	 * Find or create bank account.
	 */
	private function findOrCreateBankAccount(string $bankAccountNumber, ?string $bankName = null, ?string $swift = null, ?string $sellerName = null): ?int
	{
		$queryGenerator = new QueryGenerator('BankAccounts');
		$queryGenerator->permissions = false;
		$queryGenerator->setFields(['id']);
		$queryGenerator->addCondition('account_number', $bankAccountNumber, 'e');
		$recordId = $queryGenerator->createQuery()->scalar();

		if ($recordId) {
			return (int) $recordId;
		}

		$bankAccountRecord = \Vtiger_Record_Model::getCleanInstance('BankAccounts');

		$accountName = ($sellerName ? $sellerName . ' ' : '') . $bankAccountNumber;
		$bankAccountRecord->set('name', $accountName);
		$bankAccountRecord->set('account_number', $bankAccountNumber);

		if ($bankName) {
			$bankAccountRecord->set('bank_name', $bankName);
		}

		if ($swift) {
			$bankAccountRecord->set('swift', $swift);
		}

		$bankAccountRecord->save();

		return $bankAccountRecord->getId();
	}
}
