<?php

/**
 * KSeF XmlDeserializer Service class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Service\Reader;

use App\Integrations\KSeF\Exception\KSeFXmlException;
use App\Integrations\KSeF\KSeF;
use App\Integrations\KSeF\Model\Reader\XmlJob;
use App\Integrations\KSeF\Repository\CertificateRepository;
use App\Integrations\KSeF\Service\Reader\RecordBuilder\InvoiceCostBuilder;

/**
 * KSeF XmlDeserializer Service class.
 */
final class XmlDeserializerService
{
	/**
	 * Deserializes XML job into invoice record.
	 *
	 * @param XmlJob $job
	 * @param int    $settingId
	 *
	 * @return bool result
	 */
	public function deserialize(XmlJob $job, int $settingId): bool
	{
		try {
			$result = false;
			$payload = $job->getXmlPayload();
			if (empty($payload)) {
				throw new KSeFXmlException('Empty XML payload');
			}
			if (empty($job->getModule())) {
				throw new KSeFXmlException('Module name is empty');
			}

			libxml_use_internal_errors(true);
			$cleanedPayload = $this->removeNamespaces($payload);
			$xml = simplexml_load_string($cleanedPayload, \SimpleXMLElement::class, LIBXML_NONET | LIBXML_NOCDATA);
			if (!$xml) {
				$errors = array_map(
					static fn (\LibXMLError $e) => trim($e->message),
					libxml_get_errors()
				);
				libxml_clear_errors();

				throw new KSeFXmlException('Invalid XML: ' . implode('; ', $errors));
			}

			$companyId = (new CertificateRepository())->findById($settingId)->getCompanyId();
			$builder = new InvoiceCostBuilder($job->getModule(), $companyId);
			if ($builder->checkIfInvoiceExists($xml)) {
				KSeF::getLogger()->notice('Invoice already exists', ['params' => ['xmlJobName' => $job->getXmlName()]]);
			} else {
				$invoiceRecord = $builder->build($xml, $job->getXmlName());
				$invoiceRecord->save();
			}
			$result = true;
		} catch (\Throwable $t) {
			KSeF::getLogger()->error($t->getMessage(), ['e' => $t, 'params' => ['xmlJobId' => $job->getXmlName()]]);
		}

		return $result;
	}

	/**
	 * Remove XML namespaces to allow simple XPath queries.
	 *
	 * @param string $xmlContent
	 *
	 * @return string
	 */
	private function removeNamespaces(string $xmlContent): string
	{
		$xmlContent = preg_replace('/\sxmlns[^=]*="[^"]*"/i', '', $xmlContent);
		return preg_replace('/(<\/?)\w+:(\w+)/', '$1$2', $xmlContent);
	}
}
