<?php

/**
 * Provider for login with LDAP.
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 * @author Antoni Kiszka <a.kiszka@yetiforce.com>
 */

namespace App\Integrations\UserAuth;

use App\Authenticator\Password;
use App\Encryptions\Settings;
use App\Exceptions\AppException;
use App\Exceptions\Unauthorized;
use App\Log;
use App\Purifier;
use App\Session;

/**
 * LDAP login provider - class.
 */
class LDAP extends Password
{
	/** {@inheritdoc} */
	protected $label = 'LDAP';

	/** {@inheritdoc} */
	protected $icon = 'yfi-ldap';

	/** {@inheritdoc} */
	protected $additionalInfo = 'LBL_LDAP_RECOMMENDED_INFO';

	/** {@inheritdoc} */
	protected $editFieldNames = ['server', 'starttls', 'ldap_bind_dn', 'ldap_bind_password', 'ldap_user_filter', 'ldap_base_dn', 'ldap_user_attribute', 'ldap_timeout'];

	/** {@inheritdoc} */
	protected $encryptedFields = ['ldap_bind_password'];

	public function __construct()
	{
		parent::__construct();
		$config = $this->get('config');
		$encryption = Settings::getInstance();
		foreach ($this->encryptedFields as $field) {
			if (($secret = $config[$field] ?? '') && $encryption->isActive()) {
				$secret = $encryption->decrypt($secret);
				if (false === $secret) {
					throw new AppException('ERR_IMPOSSIBLE_DECRYPT');
				}
				$config[$field] = $secret;
				$this->set('config', $config);
			}
		}
	}

	/** {@inheritdoc} */
	public function verifyPassword(#[\SensitiveParameter] string &$password, bool $falsify = false): bool
	{
		if ($falsify || !$this->isActive()) {
			return parent::verifyPassword($password, $falsify);
		}
		$result = false;

		try {
			$result = $this->connect($password);
		} catch (\Throwable $e) {
			Log::error($e->getMessage(), 'UserAuthentication');
		}

		return $result;
	}

	/** {@inheritdoc} */
	public function preProcess()
	{
		Session::set('UserAuthMethod', 'LDAP');
	}

	/** {@inheritdoc} */
	public function getFieldInstanceByName(string $name): ?\Vtiger_Field_Model
	{
		$moduleName = 'Settings:UserAuth';
		$field = ['uitype' => 1, 'column' => $name, 'name' => $name, 'displaytype' => 1, 'typeofdata' => 'V~M', 'presence' => 0, 'isEditableReadOnly' => false];
		switch ($name) {
			case 'server':
				$field['label'] = 'LBL_LDAP_SERVER';
				$field['purifyType'] = Purifier::URL;
				$field['maximumlength'] = '3,253';
				$field['fieldvalue'] = $this->has($name) ? $this->get($name) : '';
				$field['validator'] = [['name' => 'Url']];
				$field['tooltip'] = 'LBL_LDAP_SERVER_DESC';
				break;
			case 'starttls':
				$field['label'] = 'LBL_LDAP_STARTTLS';
				$field['uitype'] = '56';
				$field['maximumlength'] = '8';
				$field['typeofdata'] = 'C~O';
				$field['purifyType'] = Purifier::BOOL;
				$field['fieldvalue'] = $this->has($name) ? $this->get($name) : '';
				$field['separator'] = true;
				break;
			case 'ldap_bind_dn':
				$field['label'] = 'LBL_LDAP_BIND_DN';
				$field['maximumlength'] = '255';
				$field['typeofdata'] = 'V~O';
				$field['purifyType'] = Purifier::TEXT;
				$field['fieldvalue'] = $this->has($name) ? $this->get($name) : '';
				$field['tooltip'] = 'LBL_LDAP_BIND_DN_DESC';
				break;
			case 'ldap_bind_password':
				$field['label'] = 'LBL_LDAP_BIND_PASSWORD';
				$field['maximumlength'] = '255';
				$field['typeofdata'] = 'V~O';
				$field['uitype'] = 99;
				$field['purifyType'] = 'raw';
				$field['fieldvalue'] = $this->has($name) ? $this->get($name) : '';
				break;
			case 'ldap_user_filter':
				$field['label'] = 'LBL_LDAP_USER_FILTER';
				$field['maximumlength'] = '255';
				$field['typeofdata'] = 'V~O';
				$field['purifyType'] = Purifier::TEXT;
				$field['fieldvalue'] = $this->has($name) ? $this->get($name) : '';
				$field['tooltip'] = 'LBL_LDAP_USER_FILTER_DESC';
				$field['separator'] = true;
				break;
			case 'ldap_base_dn':
				$field['label'] = 'LBL_LDAP_BASE_DN';
				$field['maximumlength'] = '255';
				$field['purifyType'] = Purifier::TEXT;
				$field['fieldvalue'] = $this->has($name) ? $this->get($name) : '';
				$field['tooltip'] = 'LBL_LDAP_BASE_DN_DESC';
				break;
			case 'ldap_user_attribute':
				$field['label'] = 'LBL_LDAP_USER_ATTRIBUTE';
				$field['maximumlength'] = '1,64';
				$field['purifyType'] = Purifier::TEXT;
				$field['uitype'] = 1;
				$field['fieldvalue'] = $this->has($name) ? $this->get($name) : '';
				$field['defaultvalue'] = 'uid';
				$field['tooltip'] = 'LBL_LDAP_USER_ATTRIBUTE_DESC';
				break;
			case 'ldap_timeout':
				$field['label'] = 'LBL_LDAP_TIMEOUT';
				$field['maximumlength'] = '1,30';
				$field['typeofdata'] = 'I~M';
				$field['purifyType'] = Purifier::INTEGER;
				$field['uitype'] = 7;
				$field['fieldvalue'] = $this->has($name) ? $this->get($name) : '';
				$field['defaultvalue'] = '5';
				break;
			default:
				$field = [];
				break;
		}

		return $field ? \Vtiger_Field_Model::init($moduleName, $field, $name) : null;
	}

	/**
	 * Connect to LDAP server and verify user credentials.
	 *
	 * @param string $password
	 *
	 * @return bool
	 */
	private function connect(#[\SensitiveParameter] string &$password): bool
	{
		$pwd = $password;
		$password = '[REDACTED]';

		if (!\function_exists('ldap_escape')) {
			throw new AppException('Error LDAP authentication: php extension ldap is not installed.');
		}

		$uri = $this->get('server');
		$starttls = $this->get('starttls');
		$timeout = $this->get('ldap_timeout') ?: 5;
		$baseDn = $this->get('ldap_base_dn');
		$userAttribute = $this->get('ldap_user_attribute') ?: 'uid';
		$userFilter = $this->get('ldap_user_filter');
		$userLogin = $this->userModel->getDetail('user_name');
		$login = ldap_escape($userLogin, '', LDAP_ESCAPE_DN);

		$ds = ldap_connect($uri);
		if (!$ds) {
			throw new AppException('Error LDAP authentication: Could not connect to LDAP server: ' . $uri);
		}

		ldap_set_option($ds, LDAP_OPT_PROTOCOL_VERSION, 3);
		ldap_set_option($ds, LDAP_OPT_REFERRALS, 0);
		ldap_set_option($ds, LDAP_OPT_TIMELIMIT, 5);
		ldap_set_option($ds, LDAP_OPT_TIMEOUT, $timeout);
		ldap_set_option($ds, LDAP_OPT_NETWORK_TIMEOUT, $timeout);

		if ((1 === (int) $starttls) && !ldap_start_tls($ds)) {
			throw new AppException('Error LDAP authentication: STARTTLS initialization failed for LDAP server: ' . $uri . ' |' . ldap_errno($ds) . '|' . ldap_error($ds));
		}

		if ($bindDn = $this->get('ldap_bind_dn')) {
			$bindPassword = $this->get('ldap_bind_password');
			$bind = @ldap_bind($ds, $bindDn, $bindPassword);
			if ($bind) {
				$userDn = $this->getUserDn($ds, $baseDn, $userAttribute, $userLogin, $userFilter);
			} else {
				throw new Unauthorized("Error LDAP authentication: Could not bind to LDAP server with Bind DN: {$bindDn} - |" . ldap_errno($ds) . '|' . ldap_error($ds));
			}
		} elseif (ldap_bind($ds)) {
			$userDn = $this->getUserDn($ds, $baseDn, $userAttribute, $userLogin, $userFilter);
		} else {
			$userDn = \sprintf('%s=%s,%s', $userAttribute, $login, $baseDn);
		}

		$bind = $userDn && ldap_bind($ds, $userDn, $pwd);
		if (!$bind) {
			throw new Unauthorized('Error LDAP authentication: Could not bind to LDAP server with User DN: ' . $userDn . ' - |' . ldap_errno($ds) . '|' . ldap_error($ds));
		}
		ldap_unbind($ds);

		return $bind;
	}

	/**
	 * Get user DN from LDAP.
	 *
	 * @param resource    $ldapConn
	 * @param string      $baseDn
	 * @param string      $userAttr
	 * @param string      $userLogin
	 * @param string|null $userFilter
	 *
	 * @return string|null
	 */
	private function getUserDn($ldapConn, string $baseDn, string $userAttr, string $userLogin, ?string $userFilter = null): ?string
	{
		$login = ldap_escape($userLogin, '', LDAP_ESCAPE_FILTER);
		$userFilterPart = \sprintf('(%s=%s)', $userAttr, $login);
		$userFilter = trim($userFilter ?? '');

		if (!empty($userFilter)) {
			if (preg_match('/[&|]/', $userFilter)) {
				throw new AppException('Error LDAP authentication: Unsupported characters in user filter: ' . $userFilter);
			}
			if ('(' === $userFilter[0] && ')' === $userFilter[\strlen($userFilter) - 1]) {
				$userFilter = substr($userFilter, 1, -1);
			}
			$filter = \sprintf('(&(%s)%s)', $userFilter, $userFilterPart);
		} else {
			$filter = $userFilterPart;
		}

		$search = ldap_search($ldapConn, $baseDn, $filter, ['dn']);
		if (!$search) {
			throw new Unauthorized('Error LDAP authentication: LDAP search failed for filter: ' . $filter . ' - |' . ldap_errno($ldapConn) . '|' . ldap_error($ldapConn));
		}

		$entries = ldap_get_entries($ldapConn, $search);
		if (0 === $entries['count']) {
			throw new Unauthorized('Error LDAP authentication: No entries found for filter: ' . $filter);
		}

		return $entries[0]['dn'] ?? null;
	}
}
