<?php
/**
 * Inventory currency param file.
 *
 * @package \App\Inventory
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mateusz Slominski <m.slominski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Inventory\Currency;

use App\Exceptions\AppException;
use App\Json;

/**
 * Inventory currency param class.
 */
final class Param implements \JsonSerializable
{
	/**
	 * @var array Conversion entries
	 */
	private array $entries = [];

	/**
	 * Create Param object from json.
	 *
	 * @param null|string $json
	 *
	 * @throws AppException
	 *
	 * @return self
	 */
	public static function fromJson(?string $json): self
	{
		if (null === $json || '' === $json) {
			return new self();
		}

		$decoded = Json::decode($json);
		if (null === $decoded || '' === $decoded) {
			throw new AppException('Cannot decode, invalid JSON');
		}

		if (count($decoded) === 0) {
			return new self();
		}

		return self::fromArray($decoded);
	}

	/**
	 * Create param object from array.
	 *
	 * @param array $array
	 *
	 * @return self
	 */
	public static function fromArray(array $array): self
	{
		$self = new self();

		foreach ($array as $currencyId => $conversions) {
			if (\count($conversions) > 0) {
				$self->set($currencyId, VatConversion::fromArray($conversions));
			}
			if (isset($conversions['cit'])) {
				$self->set($currencyId, CitConversion::fromArray($conversions['cit']));
			}
		}

		return $self;
	}

	/** {@inheritDoc} */
	public function jsonSerialize(): array
	{
		$result = [];
		foreach ($this->entries as $currencyId => $entry) {
			$result[$currencyId] = $this->getVatFor($currencyId)->toArray();
			if (isset($entry['cit'])) {
				$result[$currencyId]['cit'] = $this->getCitFor($currencyId)->toArray();
			}
		}

		return $result;
	}

	/**
	 * Set conversion data for given currency.
	 *
	 * @param int        $currencyId
	 * @param Conversion $conversion
	 *
	 * @return void
	 */
	public function set(int $currencyId, Conversion $conversion): void
	{
		$this->entries[$currencyId][$conversion::getType()] = $conversion;
	}

	/**
	 * Get VAT conversion data for given currency.
	 *
	 * @param int $currencyId
	 *
	 * @return VatConversion|null
	 */
	public function getVatFor(int $currencyId): ?VatConversion
	{
		return $this->entries[$currencyId]['vat'] ?? null;
	}

	/**
	 * Get CIT conversion data for given currency.
	 *
	 * @param int $currencyId
	 *
	 * @return CitConversion|null
	 */
	public function getCitFor(int $currencyId): ?CitConversion
	{
		return $this->entries[$currencyId]['cit'] ?? null;
	}
}
