<?php

/**
 * Address finder Google file.
 *
 * @see https://maps.googleapis.com Documentation  of Google Geocoding API
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */

namespace App\Map\Address;

/**
 * Address finder Google class.
 */
class GoogleGeocode extends Base
{
	/** {@inheritdoc} */
	public $docUrl = 'https://code.google.com/apis/console/?noredirect';

	/** {@inheritdoc} */
	public $customFields = [
		'key' => [
			'validator' => [['name' => 'AlphaNumeric']],
			'uitype' => 99,
			'label' => 'LBL_KEY',
			'purifyType' => \App\Purifier::ALNUM_EXTENDED,
			'maximumlength' => '200',
			'typeofdata' => 'V~M',
			'tooltip' => 'LBL_KEY_PLACEHOLDER',
		],
	];
	/**
	 * API Address to retrieve data.
	 *
	 * @var string
	 */
	protected static $url = 'https://maps.googleapis.com/maps/api/geocode/';

	/** {@inheritdoc} */
	public function find(string $value): array
	{
		if (empty($value) || !\App\RequestUtil::isNetConnection()) {
			return [];
		}
		$url = static::$url . 'json?' . http_build_query([
			'address' => $value,
			'key' => $this->config['key'],
			'language' => \App\Language::getShortLanguageName(),
		]);
		\App\Log::beginProfile("GET|GoogleGeocode::find|{$url}", __NAMESPACE__);
		$response = \App\RequestHttp::getClient()->get($url, [
			'http_errors' => false,
		]);
		\App\Log::endProfile("GET|GoogleGeocode::find|{$url}", __NAMESPACE__);

		$body = \App\Json::decode($response->getBody()->getContents());
		if (200 != $response->getStatusCode() || !empty($body['error_message'])) {
			$this->error = $body['error_message'] ?? $response->getReasonPhrase();
			\App\Log::warning('Error: ' . $url . ' | ' . $response->getStatusCode() . ' ' . $response->getReasonPhrase(), __CLASS__);
			return [];
		}
		$rows = [];
		if (isset($body['status'], $body['results'][0])) {
			foreach ($body['results'] as $row) {
				$address = $this->parse($row['address_components']);
				$rows[] = [
					'label' => $row['formatted_address'],
					'address' => $address,
					'coordinates' => [
						'lat' => $row['geometry']['location']['lat'],
						'lon' => $row['geometry']['location']['lng']
					],
					'countryCode' => strtolower($address['addresslevel1'][1] ?? ''),
				];
			}
		}
		return $rows;
	}

	/**
	 * Parse response.
	 *
	 * @param array $rows
	 *
	 * @return string[]
	 */
	private function parse(array $rows): array
	{
		$address = [];
		foreach ($rows as $row) {
			switch ($row['types'][0]) {
				case 'street_number':
					if (false !== strpos($row['long_name'], '/')) {
						[$address['buildingnumber'], $address['localnumber']] = explode('/', $row['long_name'], 2);
					} else {
						$address['buildingnumber'] = $row['long_name'];
					}
					break;
				case 'subpremise':
					$address['localnumber'] = $row['long_name'];
					break;
				case 'route':
					$address['addresslevel8'] = $row['long_name'];
					break;
				case 'postal_code':
					if (empty($row['types'][1])) {
						$address['addresslevel7'] = $row['long_name'];
					}
					break;
				case 'neighborhood':
					$address['addresslevel6'] = $row['long_name'];
					break;
				case 'sublocality':
					$address['addresslevel6'] = $row['long_name'];
					break;
				case 'locality':
					$address['addresslevel5'] = $row['long_name'];
					break;
				case 'administrative_area_level_3':
					$address['addresslevel4'] = $row['long_name'];
					break;
				case 'administrative_area_level_2':
					$address['addresslevel3'] = $row['long_name'];
					break;
				case 'administrative_area_level_1':
					$address['addresslevel2'] = $row['long_name'];
					break;
				case 'country':
					$address['addresslevel1'] = [$row['long_name'] ?? '', strtoupper($row['short_name'] ?? '')];
					break;
				default:
					break;
			}
		}
		return $address;
	}
}
