<?php

/**
 * Address finder OpenCageGeocoder file.
 *
 * @see       https://geocoder.opencagedata.com/api Documentation  of OpenCage Geocoder API
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */

namespace App\Map\Address;

/**
 * Address finder OpenCageGeocoder class.
 */
class OpenCageGeocoder extends Base
{
	/** {@inheritdoc} */
	public $customFields = [
		'country_codes' => [
			'uitype' => 1,
			'label' => 'LBL_COUNTRY_CODES',
			'purifyType' => \App\Purifier::TEXT,
			'maximumlength' => '100',
			'typeofdata' => 'V~O',
			'tooltip' => 'LBL_COUNTRY_CODES_PLACEHOLDER',
			'link' => [
				'title' => 'LBL_COUNTRY_CODES_INFO',
				'url' => 'https://wikipedia.org/wiki/List_of_ISO_3166_country_codes',
			]
		],
		'key' => [
			'validator' => [['name' => 'AlphaNumeric']],
			'uitype' => 99,
			'label' => 'LBL_KEY',
			'purifyType' => \App\Purifier::ALNUM_EXTENDED,
			'maximumlength' => '200',
			'typeofdata' => 'V~M',
			'tooltip' => 'LBL_KEY_PLACEHOLDER',
		],
	];
	/** {@inheritdoc} */
	public $docUrl = 'https://opencagedata.com/api/';
	/** @var string API Address to retrieve data. */
	protected static $url = 'https://api.opencagedata.com/geocode/v1/';

	/** {@inheritdoc} */
	public function find(string $value): array
	{
		if (empty($value) || !\App\RequestUtil::isNetConnection()) {
			return [];
		}
		$params = [
			'q' => $value,
			'pretty' => 1,
			'limit' => \App\Map\Address::getConfig()['global']['result_num'],
			'key' => $this->config['key'],
			'language' => \App\Language::getLanguage(),
		];
		if (!empty($this->config['country_codes'])) {
			$params['countrycodes'] = $this->config['country_codes'];
		}
		$url = static::$url . 'json?' . http_build_query($params);
		\App\Log::beginProfile("GET|OpenCageGeocoder::find|{$url}", __NAMESPACE__);
		$response = \App\RequestHttp::getClient()->get($url, [
			'http_errors' => false,
		]);
		\App\Log::endProfile("GET|OpenCageGeocoder::find|{$url}", __NAMESPACE__);
		$body = \App\Json::decode($response->getBody());
		if (200 !== $response->getStatusCode()) {
			$this->error = $body['status']['message'] ?? $response->getReasonPhrase();
			\App\Log::warning('Error: ' . $url . ' | ' . $response->getStatusCode() . ' ' . $response->getReasonPhrase(), __CLASS__);
			return [];
		}
		$rows = [];
		if (!empty($body['total_results'])) {
			$mainMapping = \App\Config::component('AddressFinder', 'remappingOpenCage');
			if (!\is_callable($mainMapping)) {
				$mainMapping = [$this, 'parseRow'];
			}
			$countryMapping = \App\Config::component('AddressFinder', 'remappingOpenCageForCountry');
			foreach ($body['results'] as $row) {
				$mappingFunction = $mainMapping;
				if (isset($row['components']['country'], $countryMapping[$row['components']['country']])) {
					$mappingFunction = $countryMapping[$row['components']['country']];
				}
				$rows[] = [
					'label' => $row['formatted'],
					'address' => \call_user_func_array($mappingFunction, [$row]),
					'coordinates' => [
						'lat' => $row['geometry']['lat'],
						'lon' => $row['geometry']['lng']
					],
					'countryCode' => $row['components']['country_code'] ?? '',
				];
			}
		}
		return $rows;
	}

	/**
	 * Main function to parse information about address.
	 *
	 * @param array $row
	 *
	 * @return array
	 */
	private function parseRow(array $row): array
	{
		return [
			'addresslevel1' => [$row['components']['country'] ?? '', $row['components']['ISO_3166-1_alpha-2'] ?? ''],
			'addresslevel2' => $row['components']['state'] ?? '',
			'addresslevel3' => $row['components']['state_district'] ?? $row['components']['county'] ?? '',
			'addresslevel4' => $row['components']['municipality'] ?? '',
			'addresslevel5' => $row['components']['city'] ?? $row['components']['town'] ?? $row['components']['village'] ?? '',
			'addresslevel6' => $row['components']['suburb'] ?? $row['components']['neighbourhood'] ?? $row['components']['city_district'] ?? '',
			'addresslevel7' => $row['components']['postcode'] ?? '',
			'addresslevel8' => $row['components']['road'] ?? '',
			'buildingnumber' => $row['components']['house_number'] ?? '',
			'localnumber' => $row['components']['local_number'] ?? '',
			'company_name_' => $row['components']['office'] ?? '',
		];
	}
}
