<?php
/**
 * Nominatim driver file to get coordinates.
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 *
 * @see      https://wiki.openstreetmap.org/wiki/Nominatim
 */

namespace App\Map\Coordinates;

/**
 * Nominatim driver class to get coordinates.
 */
class Nominatim extends Base
{
	/**
	 * API URL.
	 */
	private const API_URL = 'https://nominatim.openstreetmap.org';

	/**
	 * @inheritdoc
	 */
	protected string $label = 'LBL_COORDINATES_NOMINATIM';

	/**
	 * @inheritdoc
	 */
	protected string $docUrl = 'https://nominatim.org/release-docs/develop/api/Lookup';

	/**
	 * @inheritdoc
	 */
	protected array $formFields = [];

	/** {@inheritdoc} */
	public function getCoordinates(array $addressInfo, bool $onlyOne = true): ?array
	{
		if (empty($addressInfo) || !\App\RequestUtil::isNetConnection()) {
			return null;
		}
		$url = self::API_URL . '/?' . http_build_query(array_merge([
			'format' => 'json',
			'addressdetails' => 1,
			'limit' => $onlyOne ? 1 : 10,
		], $addressInfo));

		$options = ['http_errors' => false];
		\App\Log::beginProfile("GET|Nominatim::find|{$url}", __NAMESPACE__);
		$response = \App\RequestHttp::getClient()->get($url, $options);
		\App\Log::endProfile("GET|Nominatim::find|{$url}", __NAMESPACE__);
		$body = $response->getBody()->getContents();
		$body = \App\Json::isEmpty($body) ? [] : \App\Json::decode($body);

		if (200 !== $response->getStatusCode() || isset($body['error'])) {
			\App\Log::error(
				'Error: ' . $url . ' | ' . ($body['error']['message'] ?? $response->getReasonPhrase()),
				__CLASS__
			);
			return [];
		}

		$coordinates = [];
		if ($body && empty($body['error'])) {
			foreach ($body as $row) {
				$coordinate = [
					'lat' => $row['lat'],
					'lon' => $row['lon'],
				];
				if ($onlyOne) {
					$coordinates = $coordinate;
					break;
				}
				$coordinates[] = $coordinate;
			}
		}
		return $coordinates;
	}

	/** {@inheritdoc} */
	public function getCoordinatesByValue(string $value): array
	{
		return $this->getCoordinates(['q' => $value]);
	}
}
