<?php
/**
 * OpenCage driver file to get coordinates.
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 *
 * @see       https://geocoder.opencagedata.com/api Documentation  of OpenCage Geocoder API
 */

namespace App\Map\Coordinates;

/**
 * OpenCage driver class to get coordinates.
 */
class OpenCage extends Base
{
	/**
	 * Url for OpenCage API.
	 */
	private const API_URL = 'https://api.opencagedata.com/geocode/v1/json';

	/**
	 * @inheritdoc
	 */
	protected string $label = 'LBL_COORDINATES_OPEN_CAGE';

	/**
	 * @inheritdoc
	 */
	protected string $docUrl = 'https://opencagedata.com/api/';

	/**
	 * @inheritdoc
	 */
	protected array $formFields = [
		'api_key' => [
			'validator' => [['name' => 'AlphaNumeric']],
			'uitype' => 99,
			'label' => 'LBL_API_KEY',
			'purifyType' => \App\Purifier::ALNUM_EXTENDED,
			'maximumlength' => '200',
			'typeofdata' => 'V~M',
		],
	];

	/**
	 * @var string[] Address params to build address query.
	 */
	private const ADDRESS_PARAMS = ['street', 'city', 'county', 'state'];

	/** {@inheritdoc} */
	public function getCoordinates(array $addressInfo, bool $onlyOne = true): ?array
	{
		if (empty($addressInfo) || !\App\RequestUtil::isNetConnection() || empty($this->getConfig()['api_key'])) {
			return null;
		}
		$url = $this->getUrl($addressInfo) . '&limit=' . ($onlyOne ? 1 : 10);
		$options = ['http_errors' => false];
		\App\Log::beginProfile("GET|OpenCage::find|{$url}", __NAMESPACE__);
		$response = \App\RequestHttp::getClient()->get($url, $options);
		\App\Log::endProfile("GET|OpenCage::find|{$url}", __NAMESPACE__);
		$body = $response->getBody()->getContents();
		$body = \App\Json::isEmpty($body) ? [] : \App\Json::decode($body);

		if (200 !== $response->getStatusCode() || 200 != $body['status']['code']) {
			\App\Log::error(
				'Error: ' . $url . ' | ' . ($body['status']['message'] ?? $response->getReasonPhrase()),
				__CLASS__
			);
			return [];
		}

		$coordinates = [];
		if ($body && !empty($body['results'][0])) {
			foreach ($body['results'] as $row) {
				$coordinate = [
					'lat' => $row['geometry']['lat'],
					'lon' => $row['geometry']['lng']
				];
				if ($onlyOne) {
					$coordinates = $coordinate;
					break;
				}
				$coordinates[] = $coordinate;
			}
		}
		return $coordinates;
	}

	/** {@inheritdoc} */
	public function getCoordinatesByValue(string $value): array
	{
		return $this->getCoordinates(['q' => $value]);
	}

	/**
	 * Get url to Geocoding API.
	 *
	 * @param array $addressInfo
	 *
	 * @return string
	 */
	private function getUrl(array $addressInfo): string
	{
		if (isset($addressInfo['q'])) {
			$address = $addressInfo['q'];
		} else {
			$address = '';
			foreach (self::ADDRESS_PARAMS as $value) {
				if ('' !== $addressInfo[$value]) {
					$address .= $addressInfo[$value] . ',';
				}
			}
		}
		$params = [
			'q' => rtrim($address, ','),
			'language' => \App\Language::getLanguage(),
			'key' => $this->getConfig()['api_key'],
		];
		return self::API_URL . '?' . http_build_query($params);
	}
}
