<?php
/**
 * Connector to find routing. Connector based on service GraphHopper.
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 *
 * @see       https://www.graphhopper.com/developers/
 * @see       https://github.com/graphhopper/graphhopper
 */

namespace App\Map\Routing;

/**
 * Connector for service GraphHopper to get routing.
 */
class GraphHopper extends Base
{
	/**
	 * API URL.
	 */
	private const API_URL = 'https://graphhopper.com/api/1/route';

	/**
	 * @inheritdoc
	 */
	protected string $label = 'LBL_ROUTING_GRAPH_HOPPER';

	/**
	 * @inheritdoc
	 */
	protected string $docUrl = 'https://docs.graphhopper.com/#tag/Routing-API';

	/**
	 * @inheritdoc
	 */
	protected array $formFields = [
		'api_key' => [
			'validator' => [['name' => 'AlphaNumeric']],
			'uitype' => 99,
			'label' => 'LBL_API_KEY',
			'purifyType' => \App\Purifier::ALNUM_EXTENDED,
			'maximumlength' => '200',
			'typeofdata' => 'V~M',
		],
		'profile' => [
			'validator' => [['name' => 'AlphaNumeric']],
			'uitype' => 16,
			'label' => 'LBL_PROFILE',
			'purifyType' => \App\Purifier::TEXT,
			'maximumlength' => '11',
			'typeofdata' => 'V~0',
			'defaultvalue' => 'car',
			'doc' => 'https://docs.graphhopper.com/#section/Map-Data-and-Routing-Profiles',
			'picklistValues' => [
				'car' => 'car',
				'car_delivery' => 'car_delivery',
				'car_avoid_ferry' => 'car_avoid_ferry',
				'car_avoid_motorway' => 'car_avoid_motorway',
				'car_avoid_toll' => 'car_avoid_toll',
				'small_truck' => 'small_truck',
				'small_truck_delivery' => 'small_truck_delivery',
				'truck' => 'truck',
				'scooter' => 'scooter',
				'scooter_delivery' => 'scooter_delivery',
				'foot' => 'foot',
				'hike' => 'hike',
				'bike' => 'bike',
				'mtb' => 'mtb',
				'racingbike' => 'racingbike',
			],
		],
	];

	/** {@inheritdoc} */
	public function calculate(): void
	{
		if (!\App\RequestUtil::isNetConnection()) {
			return;
		}
		$options = [
			'timeout' => 120,
			'http_errors' => false,
			'json' => array_merge([
				'points' => $this->parsePoints(),
				'points_encoded' => false,
				'locale' => \App\Language::getShortLanguageName(),
			], $this->getConfig()['params']),
		];
		$url = self::API_URL . ($this->getConfig()['api_key'] ? '?key=' . $this->getConfig()['api_key'] : '');

		\App\Log::beginProfile("POST|YetiForceRouting::calculate|{$url}", __NAMESPACE__);
		$response = \App\RequestHttp::getClient()->post($url, $options);
		\App\Log::endProfile("POST|YetiForceRouting::calculate|{$url}", __NAMESPACE__);

		$body = \App\Json::decode($response->getBody()->getContents());

		if (200 !== $response->getStatusCode() || !empty($body['message'])) {
			\App\Log::error(
				'Error: ' . $url . ' | ' . ($body['message'] ?? $response->getReasonPhrase()),
				__CLASS__
			);
			return;
		}
		$this->parseResponse($body);
	}

	/** {@inheritdoc} */
	public function parsePoints(): array
	{
		$tracks = [
			[$this->start['lon'], $this->start['lat']],
		];
		if (!empty($this->indirectPoints)) {
			foreach ($this->indirectPoints as $tempLon) {
				$tracks[] = [$tempLon['lon'], $tempLon['lat']];
			}
		}
		$tracks[] = [$this->end['lon'], $this->end['lat']];
		return $tracks;
	}

	/**
	 * Parse response from API.
	 *
	 * @param array $body
	 *
	 * @return void
	 */
	private function parseResponse(array $body): void
	{
		$coordinates = [];
		$description = '';
		if (!empty($body['paths'])) {
			foreach ($body['paths'] as $path) {
				$this->distance += $path['distance'] / 1000;
				if ($this->returnDetails['duration']) {
					$this->travelTime += $path['time'] / 1000;
				}
				if ($this->returnDetails['polyline']) {
					$coordinates = array_merge($coordinates, $path['points']['coordinates']);
				}
				if ($this->returnDetails['navigationInstruction']) {
					foreach ($path['instructions'] as $instruction) {
						$description .= $instruction['text'];
						if ($instruction['distance']) {
							$description .= ' (' . (int) $instruction['distance'] . 'm)';
						}
						$description .= '<br>';
					}
				}
			}
		}
		$this->geoJson = [
			'type' => 'LineString',
			'coordinates' => $coordinates,
		];
		$this->description = $description;
	}
}
