<?php
/**
 * Overlay for Query Generator
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Łukasz Krawczyk <l.krawczyk@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Report\Builder;

use App\QueryGenerator;
use App\Report\Mapper\FilterMapper;
use App\Report\Model\Query;

/** QueryGeneratorBuilder class */
final class QueryGeneratorBuilder
{
	public function __construct(
		private readonly FilterMapper $filterMapper,
	) {
	}

	/** Adjust Query Generator for report purpose */
	public function build(Query $query, int $userId): QueryGenerator
	{
		$queryGenerator = new QueryGenerator($query->getModuleName(), $userId);

		$this->processFields($queryGenerator, $query);
		$this->processFilters($queryGenerator, $query);
		$this->processGroupBy($queryGenerator, $query);
		$this->processOrderBy($queryGenerator, $query);

		return $queryGenerator;
	}

	/** Add fields to Query Generator */
	private function processFields(QueryGenerator $queryGenerator, Query $query): void
	{
		$queryGenerator->setFields($query->getStandardFields());
		$queryGenerator->setCustomColumn($query->getCustomFields());
	}

	/** Add filters to Query Generator */
	private function processFilters(QueryGenerator $queryGenerator, Query $query): void
	{
		if (null !== $filterGroup = $query->getFilterGroup()) {
			$queryGenerator->setConditions($this->filterMapper->map($filterGroup));
		}
	}

	/** Add group by to Query Generator */
	private function processGroupBy(QueryGenerator $queryGenerator, Query $query): void
	{
		if (0 === count($query->getCustomFields())) {
			return;
		}

		foreach ($query->getGroupExpressions() as $expression) {
			$queryGenerator->setGroup($expression->getField(), $expression->getAlias());
		}
	}

	/** Add order by for Query Generator */
	private function processOrderBy(QueryGenerator $queryGenerator, Query $query): void
	{
		foreach ($query->getExpressions() as $expression) {
			if (true === $expression->isSortable()) {
				$queryGenerator->setOrder(
					$expression->getField(),
					$expression->getOrderBy(),
					$expression->getAlias(),
				);
			}
		}
	}
}
