<?php

/**
 * Module file for KSeF integration model.
 *
 * @package   Settings.Model
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

use App\Integrations\KSeF\KSeF;
use App\Integrations\KSeF\Model\Certificate\Enum\CertificateStatus;
use App\Json;
use App\Language;
use App\MultiCompany;
use App\Purifier;

/**
 * Module class for KSeF integration model.
 */
class Settings_KSeF_Module_Model extends Settings_Vtiger_Module_Model
{
	/** {@inheritdoc} */
	public $name = 'KSeF';
	/** {@inheritdoc} */
	public $baseTable = KSeF::SETTINGS_TABLE_NAME;
	/** {@inheritdoc} */
	public $baseIndex = 'id';
	/** {@inheritdoc} */
	public $listFields = [
		'name' => 'FL_NAME',
		'status' => 'FL_STATUS',
		'company' => 'FL_COMPANY',
		'date_valid_from' => 'FL_DATE_VALID_FROM',
		'date_valid_to' => 'FL_EXPIRATION_DATE',
		'contact_email' => 'FL_CONTACT_EMAIL',
	];

	/* Fields that are presented in create/edit form */
	private array $formFields = [
		'name',
		'company',
		'identifier',
		'contact_email',
		'passphrase',
		'cert',
		'key',
	];

	/** {@inheritdoc} */
	public function getCreateRecordUrl()
	{
		return 'index.php?module=' . $this->getName() . '&parent=' . $this->getParentName() . '&view=Edit';
	}

	/** {@inheritdoc} */
	public function getListViewUrl()
	{
		return 'index.php?module=' . $this->getName() . '&parent=' . $this->getParentName() . '&view=List';
	}

	/** Edit record view */
	public function getEditRecordUrl(int $recordId): string
	{
		return $this->getCreateRecordUrl() . '&record=' . $recordId;
	}

	/**
	 * Return fields for form.
	 *
	 * @return array
	 */
	public function getFormFields(): array
	{
		return $this->formFields;
	}

	public function getFieldInstanceByName(string $name): Vtiger_Field_Model
	{
		$moduleName = $this->getName(true);
		$params = [
			'uitype' => 1,
			'column' => $name,
			'name' => $name,
			'fieldvalue' => $this->get($name) ?? '',
			'displaytype' => 1,
			'typeofdata' => 'V~M',
			'presence' => 0,
			'isEditableReadOnly' => false,
			'purifyType' => Purifier::TEXT,
		];
		switch ($name) {
			case 'name':
				$params['maximumlength'] = 50;
				$params['label'] = 'FL_NAME';
				break;
			case 'status':
				$params['uitype'] = 16;
				$params['label'] = 'FL_STATUS';
				$picklistValues = [];
				foreach (CertificateStatus::cases() as $value) {
					$picklistValues[$value->getLabel()] = Language::translate($value->getLabel(), $moduleName);
				}
				$params['picklistValues'] = $picklistValues;
				break;
			case 'expiration_date':
				$params['uitype'] = 5;
				$params['label'] = 'FL_EXPIRATION_DATE';
				break;
			case 'date_valid_to':
				$params['uitype'] = 5;
				$params['label'] = 'FL_DATE_VALID_TO';
				break;
			case 'date_valid_from':
				$params['uitype'] = 5;
				$params['label'] = 'FL_DATE_VALID_FROM';
				break;
			case 'company':
				$params['uitype'] = 16;
				$params['label'] = 'FL_COMPANY';
				$params['purifyType'] = Purifier::INTEGER;
				$params['picklistValues'] = array_column(MultiCompany::getAll(), 'company_name', 'multicompanyid');
				break;
			case 'identifier':
				$params['uitype'] = 1;
				$params['label'] = 'FL_IDENTIFIER_CODE';
				$params['maximumlength'] = '10,10';
				$params['tooltip'] = 'FL_IDENTIFIER_CODE_DESC';
				$params['purifyType'] = Purifier::DIGITS;
				break;
			case 'contact_email':
				$params['uitype'] = 13;
				$params['label'] = 'FL_CONTACT_EMAIL';
				$params['tooltip'] = 'FL_CONTACT_EMAIL_DESC';
				$params['maximumlength'] = 255;
				$params['purifyType'] = Purifier::EMAIL;
				break;
			case 'passphrase':
				$params['uitype'] = 99;
				$params['label'] = 'FL_PASSPHRASE';
				$params['maximumlength'] = 65535;
				$params['purifyType'] = Purifier::RAW;
				break;
			case 'cert':
				$params['uitype'] = 330;
				$params['label'] = 'FL_CERTIFICATE';
				$params['fieldparams'] = Json::encode([
					'limit' => 1,
					'formats' => ['application/x-x509-ca-cert', 'application/x-x509-user-cert', 'application/pkix-cert', 'application/x-x509-cert', 'application/x-pem-file', 'application/pkcs7-mime', 'application/x-pkcs7-certificates', 'application/x-pkcs7-certreqresp'],
				]);
				break;
			case 'key':
				$params['uitype'] = 330;
				$params['label'] = 'FL_PRIVATE_KEY';
				$params['fieldparams'] = Json::encode([
					'limit' => 1,
				]);
				break;
			default: break;
		}

		return Settings_Vtiger_Field_Model::init($moduleName, $params, $name);
	}
}
