<?php

/**
 * MailSmtp module model class.
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Adrian Koń <a.kon@yetiforce.com>
 * @author Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */
class Settings_MailSmtp_Module_Model extends Settings_Vtiger_Module_Model
{
	public $baseTable = 's_#__mail_smtp';
	public $baseIndex = 'id';
	public $listFields = ['name' => 'LBL_NAME', 'host' => 'LBL_HOST', 'port' => 'LBL_PORT', 'username' => 'LBL_USERNAME', 'from_email' => 'LBL_FROM_EMAIL', 'default' => 'LBL_DEFAULT'];
	public $name = 'MailSmtp';

	/** @var string[] Fields name for edit view */
	public $editFields = [
		'name', 'mailer_type', 'default',
		'auth_method', 'authentication', 'individual_delivery',
		'oauth_provider', 'client_id', 'client_secret', 'redirect_uri_id', 'refresh_token',
		'secure', 'host', 'port', 'username', 'password', 'from_name', 'from_email', 'reply_to', 'priority', 'confirm_reading_to', 'organization', 'unsubscribe', 'options', 'save_send_mail', 'imap_encrypt', 'imap_host', 'imap_port', 'imap_username', 'imap_password', 'imap_folder', 'imap_validate_cert'
	];

	/**
	 * Function to get the url for default view of the module.
	 *
	 * @return string URL
	 */
	public function getDefaultUrl()
	{
		return 'index.php?module=MailSmtp&parent=Settings&view=List';
	}

	/**
	 * Function to get the url for create view of the module.
	 *
	 * @return string URL
	 */
	public function getCreateRecordUrl()
	{
		return 'index.php?module=MailSmtp&parent=Settings&view=Edit';
	}

	public static function getSmtpNames()
	{
		return (new \App\Db\Query())->select(['id', 'name'])->from('s_#__mail_smtp')->all(\App\Db::getInstance('admin'));
	}

	/**
	 * Get block icon.
	 *
	 * @param string $name
	 *
	 * @return string
	 */
	public function getBlockIcon($name): string
	{
		return '';
	}

	/**
	 * Get fields instance by name.
	 *
	 * @param string $name
	 *
	 * @return Vtiger_Field_Model
	 */
	public function getFieldInstanceByName(string $name)
	{
		$moduleName = $this->getName(true);
		$params = [];
		switch ($name) {
			case 'name':
				$params = [
					'name' => $name,
					'label' => 'LBL_NAME',
					'uitype' => 1,
					'typeofdata' => 'V~M',
					'maximumlength' => '255',
					'defaultvalue' => '',
					'purifyType' => \App\Purifier::TEXT,
					'blockLabel' => 'BL_BASE'
				];
				break;
			case 'mailer_type':
				$params = [
					'name' => $name,
					'label' => 'LBL_MAILER_TYPE',
					'uitype' => 16,
					'typeofdata' => 'V~M',
					'maximumlength' => '10',
					'purifyType' => \App\Purifier::ALNUM,
					'blockLabel' => 'BL_BASE',
					'defaultvalue' => 'smtp',
					'picklistValues' => [
						'smtp' => \App\Language::translate('LBL_SMTP', $moduleName),
						'sendmail' => \App\Language::translate('LBL_SENDMAIL', $moduleName),
						'mail' => \App\Language::translate('LBL_MAIL', $moduleName),
						'qmail' => \App\Language::translate('LBL_QMAIL', $moduleName),
					]
				];
				break;
			case 'default':
				$params = [
					'name' => $name,
					'label' => 'LBL_DEFAULT',
					'uitype' => 56,
					'typeofdata' => 'C~O',
					'maximumlength' => '1',
					'defaultvalue' => 0,
					'purifyType' => \App\Purifier::BOOL,
					'blockLabel' => 'BL_BASE'
				];
				break;
			case 'host':
				$params = [
					'name' => $name,
					'label' => 'LBL_HOST',
					'uitype' => 1,
					'typeofdata' => 'V~O',
					'maximumlength' => '240',
					'purifyType' => \App\Purifier::TEXT,
					'blockLabel' => 'BL_ACCESS_DATA'
				];
				break;
			case 'port':
				$params = [
					'name' => $name,
					'label' => 'LBL_PORT',
					'uitype' => 7,
					'typeofdata' => 'I~O',
					'maximumlength' => '0,65535',
					'purifyType' => \App\Purifier::INTEGER,
					'blockLabel' => 'BL_ACCESS_DATA'
				];
				break;
			case 'authentication':
				$params = [
					'name' => $name,
					'label' => 'LBL_AUTHENTICATION',
					'uitype' => 56,
					'typeofdata' => 'C~O',
					'maximumlength' => '1',
					'purifyType' => \App\Purifier::BOOL,
					'blockLabel' => 'BL_BASE'
				];
				break;
			case 'username':
				$params = [
					'name' => $name,
					'label' => 'LBL_USERNAME',
					'uitype' => 1,
					'typeofdata' => 'V~O',
					'maximumlength' => '255',
					'purifyType' => \App\Purifier::TEXT,
					'blockLabel' => 'BL_ACCESS_DATA'
				];
				break;
			case 'password':
				$params = [
					'name' => $name,
					'label' => 'LBL_PASSWORD',
					'uitype' => 99,
					'typeofdata' => 'V~O',
					'maximumlength' => '255',
					'purifyType' => 'raw',
					'blockLabel' => 'BL_ACCESS_DATA',
					'fromOutsideList' => true
				];
				break;
			case 'individual_delivery':
				$params = [
					'name' => $name,
					'label' => 'LBL_INDIVIDUAL_DELIVERY',
					'uitype' => 56,
					'typeofdata' => 'C~O',
					'maximumlength' => '1',
					'purifyType' => \App\Purifier::BOOL,
					'blockLabel' => 'BL_BASE',
					'tooltip' => 'LBL_INDIVIDUAL_DELIVERY_INFO'
				];
				break;
			case 'secure':
				$params = [
					'name' => $name,
					'label' => 'LBL_SECURE',
					'uitype' => 16,
					'typeofdata' => 'V~O',
					'maximumlength' => '5',
					'purifyType' => \App\Purifier::STANDARD,
					'blockLabel' => 'BL_ACCESS_DATA',
					'defaultvalue' => ''
				];
				$params['picklistValues'] = [
					'ssl' => \App\Language::translate('LBL_SSL', $moduleName),
					'tls' => \App\Language::translate('LBL_TLS', $moduleName)
				];
				break;
			case 'from_name':
				$params = [
					'name' => $name,
					'label' => 'LBL_FROM_NAME',
					'uitype' => 1,
					'typeofdata' => 'V~O',
					'maximumlength' => '255',
					'purifyType' => \App\Purifier::TEXT,
					'blockLabel' => 'BL_BASE'
				];
				break;
			case 'from_email':
				$params = [
					'name' => $name,
					'label' => 'LBL_FROM_EMAIL',
					'uitype' => 13,
					'typeofdata' => 'V~O',
					'maximumlength' => '255',
					'purifyType' => 'Email',
					'blockLabel' => 'BL_BASE'
				];
				break;
			case 'reply_to':
				$params = [
					'name' => $name,
					'label' => 'LBL_REPLY_TO',
					'uitype' => 13,
					'typeofdata' => 'V~O',
					'maximumlength' => '255',
					'purifyType' => 'Email',
					'blockLabel' => 'BL_BASE'
				];
				break;
			case 'priority':
				$params = [
					'name' => $name,
					'label' => 'LBL_MAIL_PRIORITY',
					'uitype' => 16,
					'typeofdata' => 'V~O',
					'maximumlength' => '255',
					'purifyType' => \App\Purifier::TEXT,
					'blockLabel' => 'LBL_ADDITIONAL_HEADERS',
					'defaultvalue' => '',
					'picklistValues' => [
						'normal' => \App\Language::translate('LBL_NORMAL', $moduleName),
						'non-urgent' => \App\Language::translate('LBL_NO_URGENT', $moduleName),
						'urgent' => \App\Language::translate('LBL_URGENT', $moduleName)
					]
				];
				break;
			case 'confirm_reading_to':
				$params = [
					'name' => $name,
					'label' => 'LBL_CONFIRM_READING_TO',
					'uitype' => 13,
					'typeofdata' => 'V~O',
					'maximumlength' => '255',
					'purifyType' => 'Email',
					'blockLabel' => 'LBL_ADDITIONAL_HEADERS'
				];
				break;
			case 'organization':
				$params = [
					'name' => $name,
					'label' => 'LBL_ORGANIZATION',
					'uitype' => 1,
					'typeofdata' => 'V~O',
					'maximumlength' => '255',
					'purifyType' => \App\Purifier::TEXT,
					'blockLabel' => 'LBL_ADDITIONAL_HEADERS'
				];
				break;
			case 'unsubscribe':
				$params = [
					'label' => 'LBL_UNSUBSCIBE',
					'uitype' => 33,
					'maximumlength' => '255',
					'typeofdata' => 'V~O',
					'purifyType' => \App\Purifier::TEXT,
					'createTags' => true,
					'tooltip' => 'LBL_UNSUBSCRIBE_INFO',
					'picklistValues' => [],
					'blockLabel' => 'LBL_ADDITIONAL_HEADERS'
				];
				break;
			case 'options':
				$params = [
					'label' => 'LBL_OPTIONS',
					'uitype' => 21,
					'maximumlength' => '6500',
					'typeofdata' => 'V~O',
					'purifyType' => \App\Purifier::TEXT,
					'tooltip' => 'LBL_OPTIONS_INFO',
					'blockLabel' => 'LBL_ADDITIONAL_HEADERS'
				];
				break;
			case 'save_send_mail':
				$params = [
					'name' => $name,
					'label' => 'LBL_SAVE_SEND_MAIL',
					'uitype' => 56,
					'typeofdata' => 'C~O',
					'maximumlength' => '1',
					'defaultvalue' => 0,
					'purifyType' => \App\Purifier::BOOL,
					'tooltip' => 'LBL_SAVE_SEND_MAIL_INFO',
					'blockLabel' => 'LBL_SAVE_SENT_MESSAGE'
				];
				break;
			case 'imap_encrypt':
				$params = [
					'name' => $name,
					'label' => 'LBL_SECURE',
					'uitype' => 16,
					'typeofdata' => 'V~O',
					'maximumlength' => '5',
					'purifyType' => \App\Purifier::STANDARD,
					'blockLabel' => 'LBL_SAVE_SENT_MESSAGE',
					'defaultvalue' => ''
				];
				$params['picklistValues'] = [
					'ssl' => \App\Language::translate('LBL_SSL', $moduleName),
					'tls' => \App\Language::translate('LBL_TLS', $moduleName)
				];
				break;
			case 'imap_host':
				$params = [
					'name' => $name,
					'label' => 'LBL_HOST',
					'uitype' => 17,
					'typeofdata' => 'V~O',
					'maximumlength' => '240',
					'purifyType' => \App\Purifier::URL,
					'blockLabel' => 'LBL_SAVE_SENT_MESSAGE'
				];
				break;
			case 'imap_port':
				$params = [
					'name' => $name,
					'label' => 'LBL_PORT',
					'uitype' => 7,
					'typeofdata' => 'I~O',
					'maximumlength' => '0,65535',
					'purifyType' => \App\Purifier::INTEGER,
					'blockLabel' => 'LBL_SAVE_SENT_MESSAGE'
				];
				break;
			case 'imap_validate_cert':
				$params = [
					'name' => $name,
					'label' => 'LBL_VALIDATE_CERT',
					'uitype' => 56,
					'typeofdata' => 'C~O',
					'maximumlength' => '1',
					'purifyType' => \App\Purifier::BOOL,
					'blockLabel' => 'LBL_SAVE_SENT_MESSAGE'
				];
				break;
			case 'imap_username':
				$params = [
					'name' => $name,
					'label' => 'LBL_USERNAME',
					'uitype' => 1,
					'typeofdata' => 'V~O',
					'maximumlength' => '255',
					'purifyType' => \App\Purifier::TEXT,
					'blockLabel' => 'LBL_SAVE_SENT_MESSAGE'
				];
				break;
			case 'imap_password':
				$params = [
					'name' => $name,
					'label' => 'LBL_PASSWORD',
					'uitype' => 99,
					'typeofdata' => 'V~O',
					'maximumlength' => '255',
					'purifyType' => 'raw',
					'blockLabel' => 'LBL_SAVE_SENT_MESSAGE',
					'fromOutsideList' => true
				];
				break;
			case 'imap_folder':
				$params = [
					'name' => $name,
					'label' => 'LBL_SEND_FOLDER',
					'uitype' => 1,
					'typeofdata' => 'V~M',
					'maximumlength' => '255',
					'purifyType' => \App\Purifier::TEXT,
					'blockLabel' => 'LBL_SAVE_SENT_MESSAGE'
				];
				break;
			case 'auth_method':
				$params = [
					'name' => $name,
					'label' => 'FL_AUTH_METHOD',
					'uitype' => 16,
					'typeofdata' => 'V~M',
					'maximumlength' => '50',
					'purifyType' => \App\Purifier::ALNUM,
					'blockLabel' => 'BL_BASE',
					'table' => $this->getBaseTable(),
					'defaultvalue' => 'basic',
					'picklistValues' => [
						'basic' => \App\Language::translate('LBL_BASIC_AUTH', $this->getName(true)),
						'oauth2' => \App\Language::translate('LBL_OAUTH2', $this->getName(true))
					]
				];
				break;
			case 'oauth_provider':
				$params = [
					'name' => $name,
					'label' => 'FL_OAUTH_PROVIDER',
					'uitype' => 16,
					'typeofdata' => 'V~M',
					'maximumlength' => '50',
					'purifyType' => \App\Purifier::INTEGER,
					'blockLabel' => 'BL_ACCESS_DATA',
					'defaultvalue' => '',
					'table' => $this->getBaseTable(),
					'picklistValues' => array_map(fn ($provider) => \App\Language::translate($provider->getLabel(), $this->getName(true)), \App\Integrations\OAuth::getProviders())
				];
				break;
			case 'redirect_uri_id':
				$params = [
					'name' => $name,
					'label' => 'FL_REDIRECT_URI_ID',
					'uitype' => 16,
					'typeofdata' => 'I~M',
					'maximumlength' => '2147483647',
					'purifyType' => \App\Purifier::INTEGER,
					'blockLabel' => 'BL_ACCESS_DATA',
					'tooltip' => 'LBL_REDIRECT_URI_ID_DESC',
					'defaultvalue' => '',
					'table' => $this->getBaseTable(),
					'picklistValues' => array_map(fn ($service) => $service['name'], \App\Integrations\Services::getByType(\App\Integrations\Services::OAUTH))
				];
				break;
			case 'client_id':
				$params = [
					'name' => $name,
					'label' => 'FL_CLIENT_ID',
					'uitype' => 1,
					'typeofdata' => 'V~M',
					'maximumlength' => '255',
					'purifyType' => \App\Purifier::TEXT,
					'blockLabel' => 'BL_ACCESS_DATA',
					'table' => $this->getBaseTable()
				];
				break;
			case 'client_secret':
				$params = [
					'name' => $name,
					'label' => 'FL_CLIENT_SECRET',
					'uitype' => 99,
					'typeofdata' => 'V~M',
					'maximumlength' => '255',
					'purifyType' => 'raw',
					'blockLabel' => 'BL_ACCESS_DATA',
					'fromOutsideList' => true,
					'table' => $this->getBaseTable()
				];
				break;
			case 'refresh_token':
				$params = [
					'name' => $name,
					'label' => 'FL_REFRESH_TOKEN',
					'uitype' => 99,
					'typeofdata' => 'V~M',
					'maximumlength' => '500',
					'purifyType' => 'raw',
					'blockLabel' => 'BL_ACCESS_DATA',
					'displaytype' => 2,
					'fromOutsideList' => true,
					'table' => $this->getBaseTable()
				];
				break;
			default:
				break;
		}

		return $params ? \Vtiger_Field_Model::init($this->getName(true), $params, $name) : null;
	}

	public function dependency(string $field = '')
	{
		$dependency = [
			// hide if only one condition is valid
			'imap_encrypt' => ['condition' => ['save_send_mail' => ['value' => 1, 'operator' => 'n']], 'default' => ''],
			'imap_host' => ['condition' => ['save_send_mail' => ['value' => 1, 'operator' => 'n']], 'default' => 0],
			'imap_port' => ['condition' => ['save_send_mail' => ['value' => 1, 'operator' => 'n']], 'default' => ''],
			'imap_username' => ['condition' => ['save_send_mail' => ['value' => 1, 'operator' => 'n']], 'default' => ''],
			'imap_password' => ['condition' => ['save_send_mail' => ['value' => 1, 'operator' => 'n']], 'default' => ''],
			'imap_validate_cert' => ['condition' => ['save_send_mail' => ['value' => 1, 'operator' => 'n']], 'default' => 0],
			'imap_folder' => ['condition' => ['save_send_mail' => ['value' => 1, 'operator' => 'n']], 'default' => ''],
			'oauth_provider' => ['condition' => ['auth_method' => ['value' => 'oauth2', 'operator' => 'n']], 'default' => ''],
			'client_id' => ['condition' => ['auth_method' => ['value' => 'oauth2', 'operator' => 'n']], 'default' => ''],
			'client_secret' => ['condition' => ['auth_method' => ['value' => 'oauth2', 'operator' => 'n']], 'default' => ''],
			'redirect_uri_id' => ['condition' => ['auth_method' => ['value' => 'oauth2', 'operator' => 'n']], 'default' => ''],
			'password' => ['condition' => ['auth_method' => ['value' => 'oauth2', 'operator' => 'e']], 'default' => ''],
			'refresh_token' => ['condition' => ['auth_method' => ['value' => 'hide', 'operator' => 'n']], 'default' => '']
		];

		return $field ? ($dependency[$field] ?? '') : $dependency;
	}

	/**
	 * Get structure fields.
	 *
	 * @param Settings_MailSmtp_Record_Model|null $recordModel
	 *
	 * @return array
	 */
	public function getEditViewStructure($recordModel = null): array
	{
		$structure = [];
		foreach ($this->editFields as $fieldName) {
			$fieldModel = $this->getFieldInstanceByName($fieldName);
			if ($recordModel && $recordModel->has($fieldName)) {
				$fieldModel->set('fieldvalue', $recordModel->get($fieldName));
			} else {
				$defaultValue = $fieldModel->get('defaultvalue') ?? '';
				$fieldModel->set('fieldvalue', $defaultValue);
				if ($recordModel) {
					$recordModel->set($fieldName, $defaultValue);
				}
			}
			$block = $fieldModel->get('blockLabel') ?: '';
			$structure[$block][$fieldName] = $fieldModel;
		}

		return $structure;
	}
}
