<?php

/**
 * UserAuth provider edit view class.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */

use App\Controller\ModalSettings;
use App\Exceptions\NoPermittedForAdmin;
use App\Language;
use App\Purifier;
use App\Request;
use App\UserAuth;

/**
 * UserAuth provider edit view class.
 */
class Settings_UserAuth_Edit_View extends ModalSettings
{
	/** {@inheritdoc} */
	public $modalIcon = 'yfi yfi-full-editing-view';
	/** {@inheritdoc} */
	public $modalSize = 'modal-md';

	/** {@inheritdoc} */
	public function getPageTitle(Request $request)
	{
		$moduleName = $request->getModule(false);
		$provider = UserAuth::getProviderByName($request->getByType('provider', Purifier::ALNUM));
		return Language::translate($request->isEmpty('record') ? 'LBL_USERAUTH_CREATE' : 'LBL_USERAUTH_EDIT', $moduleName) . ': ' . $provider->getLabel();
	}

	/**
	 * Check Permission.
	 *
	 * @param Request $request
	 *
	 * @throws NoPermittedForAdmin
	 */
	public function checkPermission(Request $request)
	{
		parent::checkPermission($request);
		if ($request->isEmpty('provider', true) || !UserAuth::getProviderByName($request->getByType('provider', Purifier::ALNUM)) || (!$request->isEmpty('record') && !Settings_UserAuth_Record_Model::getInstanceById($request->getInteger('record'))->has('id'))) {
			throw new NoPermittedForAdmin('LBL_PERMISSION_DENIED');
		}
	}

	/**
	 * Process.
	 *
	 * @param Request $request
	 */
	public function process(Request $request)
	{
		$qualifiedModuleName = $request->getModule(false);
		$provider = $request->getByType('provider', Purifier::ALNUM);
		if (!$request->isEmpty('record')) {
			$recordModel = Settings_UserAuth_Record_Model::getInstanceById($request->getInteger('record'));
		} else {
			$recordModel = Settings_UserAuth_Record_Model::getCleanInstance($provider);
		}

		$viewer = $this->getViewer($request);
		$viewer->assign('RECORD_MODEL', $recordModel);
		$viewer->assign('MODULE_MODEL', $recordModel->getModule());
		$viewer->assign('PROVIDER', UserAuth::getProviderByName($provider));
		$viewer->view($this->getTemplateName(), $qualifiedModuleName);
	}

	/**
	 * Template name.
	 *
	 * @return string
	 */
	public function getTemplateName(): string
	{
		return 'Edit.tpl';
	}

	/** {@inheritdoc} */
	public function getModalScripts(Request $request)
	{
		$moduleName = $request->getModule();
		return $this->checkAndConvertJsScripts([
			'modules.Settings.Vtiger.resources.Edit',
			"modules.Settings.{$moduleName}.resources.Edit"
		]);
	}
}
