<?php

use App\Cache;
use App\Config;
use App\Db;
use App\Db\Importer;
use App\Db\Importers\Base;
use App\Db\Query;
use App\Exceptions\AppException;
use App\Exceptions\Security;
use App\Fields\Currency;
use App\Fields\Double;
use App\Log;
use App\Record;
use vtlib\Utils;
use yii\db\Exception;
use yii\db\Expression;

/**
 * Basic Inventory Model Class.
 *
 * @package Model
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */
class Vtiger_Inventory_Model
{
	/** @var int Discount global mode */
	public const DISCOUT_MODE_GLOBAL = 0;

	/** @var int Discount individual mode */
	public const DISCOUT_MODE_INDIVIDUAL = 1;

	/** @var int Discount group mode */
	public const DISCOUT_MODE_GROUP = 2;

	/** @var int Tax global mode */
	public const TAX_MODE_GLOBAL = 0;

	/** @var int Tax individual mode */
	public const TAX_MODE_INDIVIDUAL = 1;

	/**
	 * Field configuration table postfix.
	 */
	private const TABLE_POSTFIX_BASE = '_invfield';

	/**
	 * Data table postfix.
	 */
	private const TABLE_POSTFIX_DATA = '_inventory';

	/**
	 * Field mapping table postfix.
	 */
	private const TABLE_POSTFIX_MAP = '_invmap';

	/**
	 * @var string
	 */
	protected $moduleName;

	/**
	 * @var Vtiger_Basic_InventoryField[] Inventory fields
	 */
	protected $fields;

	/**
	 * @var string
	 */
	protected $tableName;

	/**
	 * Gets inventory instance.
	 *
	 * @param string $moduleName
	 *
	 * @throws AppException
	 *
	 * @return self
	 */
	public static function getInstance(string $moduleName): self
	{
		if (Cache::staticHas(__METHOD__, $moduleName)) {
			$instance = Cache::staticGet(__METHOD__, $moduleName);
		} else {
			$modelClassName = Vtiger_Loader::getComponentClassName('Model', 'Inventory', $moduleName);
			$instance = new $modelClassName();
			$instance->setModuleName($moduleName);
			Cache::staticSave(__METHOD__, $moduleName, $instance);
		}
		return $instance;
	}

	/**
	 * Function returns module name.
	 *
	 * @return string
	 */
	public function getModuleName(): string
	{
		return $this->moduleName;
	}

	/**
	 * Gets table name.
	 *
	 * @param string $type
	 *
	 * @return string
	 */
	public function getTableName(string $type = self::TABLE_POSTFIX_BASE): string
	{
		if (!isset($this->tableName)) {
			$this->tableName = CRMEntity::getInstance($this->moduleName)->table_name;
		}
		return $this->tableName . $type;
	}

	/**
	 * Gets data table name.
	 *
	 * @return string
	 */
	public function getDataTableName(): string
	{
		return $this->getTableName(self::TABLE_POSTFIX_DATA);
	}

	/**
	 * Gets inventory fields.
	 *
	 * @throws AppException
	 *
	 * @return Vtiger_Basic_InventoryField[]
	 */
	public function getFields(): array
	{
		if (!isset($this->fields)) {
			$this->fields = [];
			$dataReader = (new Query())->from($this->getTableName())->indexBy('columnname')
				->orderBy(['block' => SORT_ASC, 'sequence' => SORT_ASC])->createCommand()->query();
			while ($row = $dataReader->read()) {
				$fieldModel = Vtiger_Basic_InventoryField::getInstance($this->moduleName, $row['invtype']);
				$this->setFieldData($fieldModel, $row);
				$this->fields[$row['columnname']] = $fieldModel;
			}
		}
		return $this->fields;
	}

	/**
	 * Gets inventory field model.
	 *
	 * @param string $fieldName
	 *
	 * @throws AppException
	 *
	 * @return Vtiger_Basic_InventoryField|null
	 */
	public function getField(string $fieldName): ?Vtiger_Basic_InventoryField
	{
		return $this->getFields()[$fieldName] ?? null;
	}

	/**
	 * Gets inventory field model by ID.
	 *
	 * @param int $fieldId
	 *
	 * @return Vtiger_Basic_InventoryField|null
	 */
	public function getFieldById(int $fieldId): ?Vtiger_Basic_InventoryField
	{
		$fieldModel = null;
		foreach ($this->getFields() as $field) {
			if ($fieldId === $field->getId()) {
				$fieldModel = $field;
				break;
			}
		}

		return $fieldModel;
	}

	/**
	 * Function that returns all the fields by blocks.
	 *
	 * @throws AppException
	 *
	 * @return array
	 */
	public function getFieldsByBlocks(): array
	{
		$fieldList = [];
		foreach ($this->getFields() as $fieldName => $fieldModel) {
			$fieldList[$fieldModel->get('block')][$fieldName] = $fieldModel;
		}
		return $fieldList;
	}

	/**
	 * Function that returns all the fields for given block ID.
	 *
	 * @param int $blockId
	 *
	 * @return Vtiger_Basic_InventoryField[]
	 */
	public function getFieldsByBlock(int $blockId): array
	{
		return $this->getFieldsByBlocks()[$blockId] ?? [];
	}

	/**
	 * Get syncronized fields.
	 *
	 * @return Vtiger_Basic_InventoryField[]
	 */
	public function getFieldsToSync(): array
	{
		$fieldList = [];
		foreach ($this->getFields() as $fieldName => $fieldModel) {
			if (0 === $fieldModel->get('block') || $fieldModel->isSync()) {
				$fieldList[$fieldName] = $fieldModel;
			}
		}

		return $fieldList;
	}

	/**
	 * Gets inventory fields by type.
	 *
	 * @param string $type
	 *
	 * @throws AppException
	 *
	 * @return Vtiger_Basic_InventoryField[]
	 */
	public function getFieldsByType(string $type): array
	{
		$fieldList = [];
		foreach ($this->getFields() as $fieldName => $fieldModel) {
			if ($type === $fieldModel->getType()) {
				$fieldList[$fieldName] = $fieldModel;
			}
		}
		return $fieldList;
	}

	/**
	 * Gets the field for the view.
	 *
	 * @param string $view
	 *
	 * @throws AppException
	 *
	 * @return Vtiger_Basic_InventoryField[]
	 */
	public function getFieldsForView(string $view): array
	{
		$fieldList = [];
		switch ($view) {
			case 'DetailPreview':
			case 'Detail':
				foreach ($this->getFields() as $fieldName => $fieldModel) {
					if ($fieldModel->isVisibleInDetail()) {
						$fieldList[$fieldModel->get('block')][$fieldName] = $fieldModel;
					}
				}
				break;
			default:
				break;
		}
		return $fieldList;
	}

	/**
	 * Getting summary fields name.
	 *
	 * @param bool $active
	 * @param int  $type
	 *
	 * @throws AppException
	 *
	 * @return string[]
	 */
	public function getSummaryFields(bool $active = true, int $type = Vtiger_Basic_InventoryField::SUMMARY_ON | Vtiger_Basic_InventoryField::SUMMARY_BLOCK | Vtiger_Basic_InventoryField::SUMMARY_BY_GROUP): array
	{
		$summaryFields = [];
		foreach ($this->getFields() as $name => $field) {
			if ($active ? $field->isSummaryEnabled($type) : $field->isSummary()) {
				$summaryFields[$name] = $name;
			}
		}
		return $summaryFields;
	}

	/**
	 * Sets inventory field data.
	 *
	 * @param Vtiger_Basic_InventoryField $fieldModel
	 * @param array                       $row
	 */
	public function setFieldData(Vtiger_Basic_InventoryField $fieldModel, array $row)
	{
		$fieldModel->set('id', (int) $row['id'])
			->set('columnname', $row['columnname'])
			->set('label', $row['label'])
			->set('presence', (int) $row['presence'])
			->set('defaultvalue', $row['defaultvalue'])
			->set('sequence', (int) $row['sequence'])
			->set('block', (int) $row['block'])
			->set('displaytype', (int) $row['displaytype'])
			->set('params', $row['params'])
			->set('invtype', $row['invtype'])
			->set('colspan', (int) $row['colspan']);
	}

	/**
	 * Checks if inventory field exists.
	 *
	 * @param string $fieldName
	 *
	 * @throws AppException
	 *
	 * @return bool
	 */
	public function isField(string $fieldName): bool
	{
		return isset($this->getFields()[$fieldName]);
	}

	/**
	 * Gets clean inventory field instance.
	 *
	 * @param string $type
	 *
	 * @throws AppException
	 *
	 * @return Vtiger_Basic_InventoryField
	 */
	public function getFieldCleanInstance(string $type): Vtiger_Basic_InventoryField
	{
		return Vtiger_Basic_InventoryField::getInstance($this->getModuleName(), $type);
	}

	/**
	 * Function to get data of inventory for record.
	 *
	 * @param int                      $recordId
	 * @param string                   $moduleName
	 * @param Vtiger_Paging_Model|null $pagingModel
	 *
	 * @throws AppException
	 *
	 * @return array
	 */
	public static function getInventoryDataById(int $recordId, string $moduleName, ?Vtiger_Paging_Model $pagingModel = null): array
	{
		$inventory = self::getInstance($moduleName);
		$query = (new Query())->from($inventory->getTableName(self::TABLE_POSTFIX_DATA))->indexBy('id')->where(['crmid' => $recordId]);
		if ($inventory->isField('seq')) {
			$query->orderBy(['seq' => SORT_ASC]);
		}
		if ($pagingModel) {
			$pageLimit = $pagingModel->getPageLimit();
			if (0 !== $pagingModel->get('limit')) {
				$query->limit($pageLimit + 1)->offset($pagingModel->getStartIndex());
			}
			$rows = $query->all();
			$count = \count($rows);
			if ($count > $pageLimit) {
				array_pop($rows);
				$pagingModel->set('nextPageExists', true);
			} else {
				$pagingModel->set('nextPageExists', false);
			}
			$pagingModel->calculatePageRange($count);
			return $rows;
		}
		return $query->all();
	}

	/**
	 * Save inventory field.
	 *
	 * @param Vtiger_Basic_InventoryField $fieldModel
	 *
	 * @throws Exception
	 *
	 * @return bool
	 */
	public function saveField(Vtiger_Basic_InventoryField $fieldModel): bool
	{
		$db = Db::getInstance();
		$tableName = $this->getTableName();
		if (!$fieldModel->has('sequence')) {
			$fieldModel->set('sequence', $db->getUniqueID($tableName, 'sequence', false));
		}
		if (!$fieldModel->getId() && !$fieldModel->isOnlyOne()) {
			$id = (new Query())->from($tableName)->where(['invtype' => $fieldModel->getType()])->max('id') + 1;
			$fieldModel->set('columnname', $fieldModel->getColumnName() . $id);
		}
		$transaction = $db->beginTransaction();
		try {
			$result = true;
			if (!$fieldModel->getId()) {
				$table = $this->getTableName(self::TABLE_POSTFIX_DATA);
				Utils::addColumn($table, $fieldModel->getColumnName(), $fieldModel->getDBType());
				foreach ($fieldModel->getCustomColumn() as $column => $criteria) {
					Utils::addColumn($table, $column, $criteria);
				}
				$result = $db->createCommand()->insert($tableName, $fieldModel->getData())->execute();
				$fieldModel->set('id', $db->getLastInsertID("{$tableName}_id_seq"));
			} elseif ($data = array_intersect_key($fieldModel->getData(), $fieldModel->getPreviousValue())) {
				$result = $db->createCommand()->update($tableName, $data, ['id' => $fieldModel->getId()])->execute();
			}
			$transaction->commit();
		} catch (Throwable $ex) {
			$transaction->rollBack();
			Log::error($ex->__toString());
			$result = false;
		}

		return (bool) $result;
	}

	/**
	 * Delete inventory field.
	 *
	 * @param string $fieldName
	 *
	 * @throws Exception
	 *
	 * @return bool
	 */
	public function deleteField(string $fieldName): bool
	{
		$db = Db::getInstance();
		$dbCommand = $db->createCommand();
		$transaction = $db->beginTransaction();
		$result = false;
		try {
			$fieldModel = $this->getField($fieldName);
			$columnsArray = array_keys($fieldModel->getCustomColumn());
			$columnsArray[] = $fieldName;
			if (isset($fieldModel->shared)) {
				foreach ($fieldModel->shared as $column => $columnShared) {
					if ($this->isField($columnShared) && false !== ($key = array_search($column, $columnsArray))) {
						unset($columnsArray[$key]);
					}
				}
			}
			$dbCommand->delete($this->getTableName(), ['columnname' => $fieldName])->execute();
			if ('seq' !== $fieldName) {
				foreach ($columnsArray as $column) {
					$dbCommand->dropColumn($this->getTableName(self::TABLE_POSTFIX_DATA), $column)->execute();
				}
			}
			$transaction->commit();
			$result = true;
		} catch (Throwable $ex) {
			$transaction->rollBack();
			Log::error($ex->__toString());
		}
		return $result;
	}

	/**
	 * Save sequence field.
	 *
	 * @param int[] $sequenceList
	 *
	 * @throws Exception
	 *
	 * @return int
	 */
	public function saveSequence(array $sequenceList): int
	{
		$db = Db::getInstance();
		$case = 'CASE id';
		foreach ($sequenceList as $sequence => $id) {
			$case .= " WHEN {$db->quoteValue($id)} THEN {$db->quoteValue($sequence)}";
		}
		$case .= ' END ';
		return $db->createCommand()->update($this->getTableName(), ['sequence' => new Expression($case)], ['id' => $sequenceList])->execute();
	}

	/**
	 * Retrieve list of all fields.
	 *
	 * @throws AppException
	 *
	 * @return Vtiger_Basic_InventoryField[] Fields instance
	 */
	public function getFieldsTypes(): array
	{
		$moduleName = $this->getModuleName();
		if (Cache::has(__METHOD__, $moduleName)) {
			$inventoryTypes = Cache::get(__METHOD__, $moduleName);
		} else {
			if (Config::performance('LOAD_CUSTOM_FILES')) {
				$fieldPaths[] = "custom/modules/{$moduleName}/inventoryfields/";
			}
			$fieldPaths[] = "modules/{$moduleName}/inventoryfields/";
			if ('Vtiger' !== $moduleName) {
				$fieldPaths[] = 'modules/Vtiger/inventoryfields/';
			}
			$inventoryTypes = [];
			foreach ($fieldPaths as $fieldPath) {
				if (!is_dir($fieldPath)) {
					continue;
				}
				foreach (new DirectoryIterator($fieldPath) as $object) {
					if ('php' === $object->getExtension() && 'Basic' !== ($type = $object->getBasename('.php')) && !isset($inventoryTypes[$type])) {
						$inventoryTypes[$type] = Vtiger_Basic_InventoryField::getInstance($moduleName, $type);
					}
				}
			}
			Cache::save(__METHOD__, $moduleName, $inventoryTypes);
		}
		return $inventoryTypes;
	}

	/**
	 * Gets all columns.
	 *
	 * @throws AppException
	 *
	 * @return Vtiger_Basic_InventoryField[]
	 */
	public function getAllColumns()
	{
		$columns = [];
		foreach ($this->getFields() as $name => $field) {
			$columns[$name] = $field;
			foreach (array_keys($field->getCustomColumn()) as $name) {
				$columns[$name] = $field;
			}
		}

		return $columns;
	}

	/**
	 * Function return autocomplete fields.
	 *
	 * @return array
	 */
	public function getAutoCompleteFields()
	{
		$moduleName = $this->getModuleName();
		if (Cache::has(__METHOD__, $moduleName)) {
			$fields = Cache::get(__METHOD__, $moduleName);
		} else {
			$fields = [];
			$dataReader = (new Query())->from($this->getTableName(self::TABLE_POSTFIX_MAP))->createCommand()->query();
			while ($row = $dataReader->read()) {
				$fields[$row['module']][$row['tofield']] = $row;
			}
			Cache::save(__METHOD__, $moduleName, $fields);
		}
		return $fields;
	}

	/**
	 * Function to get custom values to complete in inventory.
	 *
	 * @param string              $sourceFieldName
	 * @param Vtiger_Record_Model $recordModel
	 *
	 * @return array
	 */
	public function getCustomAutoComplete(string $sourceFieldName, Vtiger_Record_Model $recordModel)
	{
		$values = [];
		$inventoryMap = Config::module($this->getModuleName(), 'INVENTORY_ON_SELECT_AUTO_COMPLETE');
		if ($inventoryMap) {
			foreach ($inventoryMap as $fieldToComplete => $mapping) {
				if (isset($mapping[$sourceFieldName]) && method_exists($this, $mapping[$sourceFieldName])) {
					$methodName = $mapping[$sourceFieldName];
					$values[$fieldToComplete] = $this->{$methodName}($recordModel);
				}
			}
		}
		return $values;
	}

	/**
	 * Gets data from record.
	 *
	 * @param Vtiger_Record_Model $recordModel
	 *
	 * @return float
	 */
	public function getInventoryPrice(Vtiger_Record_Model $recordModel)
	{
		return $recordModel->isEmpty('sum_total') ? 0 : $recordModel->get('sum_total');
	}

	/**
	 * Function to get list elements in iventory as html code.
	 *
	 * @param Vtiger_Record_Model $recodModel
	 *
	 * @throws AppException
	 *
	 * @return string
	 */
	public function getInventoryListName(Vtiger_Record_Model $recodModel)
	{
		$field = $this->getField('name');
		$html = '<ul>';
		foreach ($recodModel->getInventoryData() as $data) {
			$html .= '<li>';
			$html .= $field->getDisplayValue($data['name']);
			$html .= '</li>';
		}
		return $html . '</ul>';
	}

	/**
	 * Get edit value.
	 *
	 * @param array  $itemData
	 * @param string $column
	 * @param string $default
	 *
	 * @return mixed
	 */
	public function getEditValue(array $itemData, string $column, $default = '')
	{
		$value = $default;
		if ($fieldModel = $this->getAllColumns()[$column] ?? null) {
			$value = $fieldModel->getEditValue($itemData, $column, $default);
		}

		return $value;
	}

	/**
	 * Gets template to purify.
	 *
	 * @throws AppException
	 *
	 * @return array
	 */
	public function getPurifyTemplate(): array
	{
		$template = [];
		foreach ($this->getFields() as $fieldModel) {
			$template += $fieldModel->getPurifyType();
		}
		return $template;
	}

	/**
	 * Get discounts configuration.
	 *
	 * @param string $key
	 *
	 * @return mixed config data
	 */
	public static function getDiscountsConfig(string $key = '')
	{
		if (Cache::has('Inventory', 'DiscountConfiguration')) {
			$config = Cache::get('Inventory', 'DiscountConfiguration');
		} else {
			$config = [];
			$dataReader = (new Query())->from('a_#__discounts_config')->createCommand(Db::getInstance('admin'))->query();
			while ($row = $dataReader->read()) {
				$name = $row['param'];
				if ('discounts' === $name) {
					$discounts = $row['value'] ? explode(',', $row['value']) : [];
					$value = array_map(static fn ($val) => (int) $val, $discounts);
				} else {
					$value = (int) $row['value'];
				}
				$config[$name] = $value;
			}
			Cache::save('Inventory', 'DiscountConfiguration', $config, Cache::LONG);
		}
		return $key ? $config[$key] : $config;
	}

	/**
	 * Get global discounts list.
	 *
	 * @return array discounts list
	 */
	public function getGlobalDiscounts()
	{
		if (Cache::has('Inventory', 'Discounts')) {
			return Cache::get('Inventory', 'Discounts');
		}
		$discounts = (new Query())->from('a_#__discounts_global')->where(['status' => 0])
			->createCommand(Db::getInstance('admin'))->queryAllByGroup(1);
		Cache::save('Inventory', 'Discounts', $discounts, Cache::LONG);
		return $discounts;
	}

	/**
	 * Get tax configuration.
	 *
	 * @return array config data
	 */
	public static function getTaxesConfig()
	{
		if (Cache::has('Inventory', 'TaxConfiguration')) {
			return Cache::get('Inventory', 'TaxConfiguration');
		}
		$config = [];
		$dataReader = (new Query())->from('a_#__taxes_config')->createCommand(Db::getInstance('admin'))->query();
		while ($row = $dataReader->read()) {
			$value = $row['value'];
			if (\in_array($row['param'], ['taxs'])) {
				$value = array_map(static fn ($val) => (int) $val, explode(',', $value));
			}
			$config[$row['param']] = \is_array($value) ? $value : (int) $value;
		}
		Cache::save('Inventory', 'TaxConfiguration', $config, Cache::LONG);
		return $config;
	}

	/**
	 * Get global tax list.
	 *
	 * @return array tax list
	 */
	public static function getGlobalTaxes()
	{
		if (Cache::has('Inventory', 'Taxes')) {
			return Cache::get('Inventory', 'Taxes');
		}
		$taxes = (new Query())->from('a_#__taxes_global')->where(['status' => 0])
			->createCommand(Db::getInstance('admin'))->queryAllByGroup(1);
		Cache::save('Inventory', 'Taxes', $taxes, Cache::LONG);
		return $taxes;
	}

	/**
	 * Get default global tax .
	 *
	 * @return array tax list
	 */
	public static function getDefaultGlobalTax()
	{
		if (Cache::has('Inventory', 'DefaultTax')) {
			return Cache::get('Inventory', 'DefaultTax');
		}
		$defaultTax = (new Query())->from('a_#__taxes_global')->where(['status' => 0])->andWhere(['default' => 1])
			->one();
		Cache::save('Inventory', 'DefaultTax', $defaultTax, Cache::LONG);
		return $defaultTax;
	}

	/**
	 * Get discount from the account.
	 *
	 * @param string $moduleName    Module name
	 * @param int    $record        Record ID
	 * @param mixed  $relatedRecord
	 *
	 * @return array
	 */
	public function getAccountDiscount($relatedRecord)
	{
		$discount = 0;
		$discountField = 'discount';
		$recordName = '';
		if (!empty($relatedRecord)) {
			$accountRecordModel = Vtiger_Record_Model::getInstanceById($relatedRecord);
			$discount = $accountRecordModel->get($discountField);
			$recordName = $accountRecordModel->getName();
		}
		return ['discount' => $discount, 'name' => $recordName];
	}

	/**
	 * Get tax from the account.
	 *
	 * @param int $relatedRecord Record ID
	 *
	 * @return array
	 */
	public function getAccountTax($relatedRecord)
	{
		$sourceModule = 'Accounts';
		$recordName = '';
		$accountTaxes = [];
		if (!empty($relatedRecord) && Record::isExists($relatedRecord, $sourceModule) && ($taxField = current(Vtiger_Module_Model::getInstance($sourceModule)->getFieldsByUiType(303))) && $taxField->isActiveField()) {
			$accountRecordModel = Vtiger_Record_Model::getInstanceById($relatedRecord, $sourceModule);
			$accountTaxes = Vtiger_Taxes_UIType::getValues($accountRecordModel->get($taxField->getName()));
			$recordName = $accountRecordModel->getName();
		}
		return ['taxes' => $accountTaxes, 'name' => $recordName];
	}

	/**
	 * Create inventory tables.
	 */
	public function createInventoryTables()
	{
		$db = Db::getInstance();
		$importer = new Base();
		$focus = CRMEntity::getInstance($this->getModuleName());
		$dataTableName = $this->getTableName(self::TABLE_POSTFIX_DATA);
		$mapTableName = $this->getTableName(self::TABLE_POSTFIX_MAP);
		$charset = Config::db('base')['charset'] ?? 'utf8mb4';
		$tables = [
			$dataTableName => [
				'columns' => [
					'id' => $importer->primaryKey(10),
					'crmid' => $importer->integer(10),
					'seq' => $importer->integer(10),
				],
				'index' => [
					["{$dataTableName}_crmid_idx", 'crmid'],
				],
				'engine' => 'InnoDB',
				'charset' => $charset,
				'foreignKey' => [
					["{$dataTableName}_crmid_fk", $dataTableName, 'crmid', $focus->table_name, $focus->table_index, 'CASCADE', null]
				]
			],
			$this->getTableName(self::TABLE_POSTFIX_BASE) => [
				'columns' => [
					'id' => $importer->primaryKey(),
					'columnname' => $importer->stringType(30)->notNull(),
					'label' => $importer->stringType(50)->notNull(),
					'invtype' => $importer->stringType(30)->notNull(),
					'presence' => $importer->smallInteger(1)->unsigned()->notNull()->defaultValue(0),
					'defaultvalue' => $importer->stringType(),
					'sequence' => $importer->integer(10)->unsigned()->notNull(),
					'block' => $importer->smallInteger(1)->unsigned()->notNull(),
					'displaytype' => $importer->smallInteger(1)->unsigned()->notNull()->defaultValue(1),
					'params' => $importer->text(),
					'colspan' => $importer->smallInteger(1)->unsigned()->notNull()->defaultValue(1),
				],
				'engine' => 'InnoDB',
				'charset' => $charset,
			],
			$mapTableName => [
				'columns' => [
					'module' => $importer->stringType(50)->notNull(),
					'field' => $importer->stringType(50)->notNull(),
					'tofield' => $importer->stringType(50)->notNull(),
				],
				'primaryKeys' => [
					["{$mapTableName}_pk", ['module', 'field', 'tofield']],
				],
				'engine' => 'InnoDB',
				'charset' => $charset,
			]];
		$base = new Importer();
		$base->dieOnError = Config::debug('SQL_DIE_ON_ERROR');
		foreach ($tables as $tableName => $data) {
			if (!$db->isTableExists($tableName)) {
				$importer->tables = [$tableName => $data];
				$base->addTables($importer);
				if (isset($data['foreignKey'])) {
					$importer->foreignKey = $data['foreignKey'];
					$base->addForeignKey($importer);
				}
			}
		}
	}

	/**
	 * Load row data by record Id.
	 *
	 * @param int   $recordId
	 * @param array $params
	 *
	 * @return array
	 */
	public function loadRowData(int $recordId, array $params = []): array
	{
		$recordModel = Vtiger_Record_Model::getInstanceById($recordId);
		$recordModuleName = $recordModel->getModuleName();
		$data = [
			'name' => $recordId,
		];
		if (!$recordModel->isEmpty('description')) {
			$data['comment1'] = $recordModel->get('description');
		}
		if (\in_array($recordModuleName, ['Products', 'Services'])) {
			$currencyId = $params['currency'] ?? Currency::getDefault()['id'];
			if (($fieldModel = $recordModel->getField('unit_price')) && $fieldModel->isActiveField()) {
				$data['price'] = $fieldModel->getUITypeModel()->getValueForCurrency($recordModel->get($fieldModel->getName()), $currencyId);
			}
			if (($fieldModel = $recordModel->getField('purchase')) && $fieldModel->isActiveField()) {
				$data['purchase'] = $fieldModel->getUITypeModel()->getValueForCurrency($recordModel->get($fieldModel->getName()), $currencyId);
			}
		}
		if ($autoCompleteField = ($this->getAutoCompleteFields()[$recordModuleName] ?? [])) {
			foreach ($autoCompleteField as $field) {
				$fieldModel = $recordModel->getField($field['field']);
				if ($fieldModel && ($fieldValue = $recordModel->get($field['field']))) {
					$data[$field['tofield']] = $fieldValue;
				}
			}
		}
		return $data;
	}

	/**
	 * Transform data.
	 *
	 * @param array $data
	 *
	 * @return array
	 */
	public function transformData(array $data): array
	{
		$set = [];
		foreach ($data as &$row) {
			$groupId = $row['groupid'] ?? null;
			if ($groupId && !isset($set[$groupId])) {
				$set[$groupId] = $groupId;
				$row['add_header'] = true;
			}
		}

		return $data;
	}

	/**
	 * Check if comment fields are empty.
	 *
	 * @param array $data
	 *
	 * @return bool
	 */
	public function isCommentFieldsEmpty(array $data)
	{
		$isEmpty = true;
		foreach ($this->getFieldsByType('Comment') as $fieldModel) {
			if ($fieldModel->isVisible() && $this->getEditValue($data, $fieldModel->getColumnName())) {
				$isEmpty = false;
				break;
			}
		}

		return $isEmpty;
	}

	/**
	 * Returns recalculated dynamic values based on provided inventory data.
	 *
	 * @param array $inventory
	 * @param bool  $userFormat
	 *
	 * @throws AppException
	 * @throws Security
	 *
	 * @return array
	 */
	public function recalculateValues(array $inventory, bool $userFormat = true): array
	{
		$inventoryData = $this->getRecalculateInventoryData($inventory, $userFormat);
		$recordModel = Vtiger_Record_Model::getCleanInstance($this->getModuleName());
		$recordModel->initInventoryData($inventoryData, $userFormat);

		$summary = $this->formatSummaryToDisplay($this->getAllSummaryValues($recordModel));
		$currencySummary = $this->getField('currency')?->getCurrencyConversationSummary($recordModel) ?? [];
		$taxSummary = $this->getField('tax')?->getTaxSummary($recordModel) ?? [];

		return [
			'inventory' => $this->getInventoryEditData($recordModel),
			'summary' => $summary['total'],
			'tax_summary' => $this->formatSummaryToDisplay($taxSummary),
			'groups_summary' => $summary['groups'],
			'discount_summary' => $summary['total']['sum_discount'] ?? Double::formatToDisplay('0.00'),
			'currency_summary' => $this->formatSummaryToDisplay($currencySummary),
		];
	}

	/**
	 * Returns recalculated invenotry values based on provided inventory data.
	 *
	 * @param array $inventory
	 * @param bool  $userFormat
	 *
	 * @return array
	 */
	public function getRecalculateInventoryData(array $inventory, bool $userFormat = true): array
	{
		$inventoryData = [];
		foreach ($inventory as $inventoryKey => $inventoryItem) {
			foreach ($this->getFields() as $columnName => $fieldModel) {
				if (\in_array($fieldModel->getColumnName(), ['currency', 'discount_aggreg', 'name', 'qty', 'price', 'purchase', 'discountmode', 'grouplabel', 'taxmode'])) {
					$item[$columnName] = $inventoryItem[$columnName] ?? ($userFormat ? $fieldModel->getEditValue($inventoryItem) : $fieldModel->getDBValue($fieldModel->getEditValue($inventoryItem)));
				} elseif (!$fieldModel->isCalculated() && isset($inventoryItem[$columnName])) {
					$item[$columnName] = $inventoryItem[$columnName];
				}
				foreach (array_keys($fieldModel->getCustomColumn()) as $columnName) {
					if (isset($inventoryItem[$columnName])) {
						$item[$columnName] = $inventoryItem[$columnName];
					}
				}
			}

			$item['id'] = $inventoryKey;
			$inventoryData[$inventoryKey] = $item;
		}

		return $inventoryData;
	}

	/**
	 * Returns fields summaries of whole inventory and for each group.
	 *
	 * @param Vtiger_Record_Model $model
	 *
	 * @throws AppException
	 *
	 * @return array[] {'total' => total summary value, 'groups' => values groupped by groupid}
	 */
	public function getAllSummaryValues(Vtiger_Record_Model $model): array
	{
		$inventoryData = $model->getInventoryData();
		$groupsIds = array_filter(array_unique(array_column($inventoryData, 'groupid')));

		$groups = [];
		$total = [];
		foreach ($this->getSummaryFields() as $fieldName) {
			$fieldModel = $this->getField($fieldName);
			$value = $fieldModel->getSummaryValuesFromData($inventoryData);
			$total['sum_' . $fieldName] = $fieldModel->roundDecimal($value);

			foreach ($groupsIds as $groupId) {
				$value = $fieldModel->getSummaryValuesFromData($inventoryData, $groupId);
				$groups[$groupId]['sum_' . $fieldName] = $fieldModel->roundDecimal($value);
			}
		}

		return [
			'total' => $total,
			'groups' => $groups,
		];
	}

	/**
	 * Returns correction tax summary with relation to base inventory for each tax group.
	 *
	 * @param Vtiger_Record_Model $baseModel       base inventory model
	 * @param Vtiger_Record_Model $correctionModel correction inventory model
	 *
	 * @throws AppException
	 *
	 * @return array
	 */
	public function getCorrectionTaxSummary(Vtiger_Record_Model $baseModel, Vtiger_Record_Model $correctionModel): array
	{
		$baseInventory = self::getInstance($baseModel->getModuleName());

		if (!$this->isField('tax') || !$baseInventory->isField('tax')) {
			return [];
		}

		$taxField = $this->getField('tax');
		$baseTaxSummary = $baseInventory->getField('tax')->getTaxSummary($baseModel);
		$correctionTaxSummary = $taxField->getTaxSummary($correctionModel);

		$groups = [];
		foreach ($correctionTaxSummary['groups'] as $key => $value) {
			if (isset($baseTaxSummary['groups'][$key])) {
				$groups[$key] = $taxField->roundDecimal($value - $baseTaxSummary['groups'][$key]);
				unset($baseTaxSummary['groups'][$key]);
			} else {
				$groups[$key] = $taxField->roundDecimal($value);
			}
		}

		if (\count($baseTaxSummary['groups']) > 0) {
			foreach ($baseTaxSummary['groups'] as $key => $value) {
				$groups[$key] = $taxField->roundDecimal(0 != $value ? -1 * $value : 0);
			}
		}

		return [
			'groups' => $groups,
			'total' => $taxField->roundDecimal($correctionTaxSummary['total'] - $baseTaxSummary['total']),
		];
	}

	/**
	 * Returns gross difference between correction and base inventory model.
	 *
	 * @param Vtiger_Record_Model $baseModel
	 * @param Vtiger_Record_Model $correctionModel
	 *
	 * @throws AppException
	 *
	 * @return float
	 */
	public function getCorrectionGrossSummary(Vtiger_Record_Model $baseModel, Vtiger_Record_Model $correctionModel): float
	{
		return $this->getField('gross')->roundDecimal($correctionModel->get('sum_gross') - $baseModel->get('sum_gross'));
	}

	/**
	 * Returns inventory data formatted for display.
	 *
	 * @param Vtiger_Record_Model $model
	 *
	 * @throws AppException
	 *
	 * @return array formatted inventory data
	 */
	public function getInventoryEditData(Vtiger_Record_Model $model)
	{
		$result = [];

		$fields = $this->getFields();
		foreach ($model->getInventoryData() as $key => $items) {
			foreach ($items as $column => $value) {
				if (!isset($fields[$column])) {
					$result[$key][$column] = $value;
				} else {
					$result[$key][$column] = $fields[$column]->getEditValue($items);
				}
			}
		}

		return $result;
	}

	/**
	 * Formats summary values for display.
	 *
	 * @param array $summary
	 *
	 * @return array
	 */
	public function formatSummaryToDisplay(array $summary): array
	{
		$result = [
			'groups' => [],
			'total' => []
		];

		if (isset($summary['groups'])) {
			foreach ($summary['groups'] as $group => $groupData) {
				$result['groups'][$group] = \is_array($groupData)
					? array_map(static fn ($value) => Double::formatToDisplay($value), $groupData)
					: Double::formatToDisplay($groupData);
			}
		}

		if (isset($summary['total'])) {
			if (\is_array($summary['total'])) {
				foreach ($summary['total'] as $key => $value) {
					$result['total'][$key] = Double::formatToDisplay($value);
				}
			} else {
				$result['total'] = Double::formatToDisplay($summary['total']);
			}
		}

		return $result;
	}

	/**
	 * Returns Currency inventory details.
	 *
	 * @param Vtiger_Record_Model $model
	 *
	 * @throws AppException
	 *
	 * @return array
	 */
	public function getCurrency(Vtiger_Record_Model $model): array
	{
		$inventoryRows = $model->getInventoryData();
		$firstRow = current($inventoryRows);
		if ($this->isField('currency')) {
			$currencyId = !empty($firstRow) && null !== $firstRow['currency']
				? $firstRow['currency']
				: Vtiger_Util_Helper::getBaseCurrency()['id'];

			return Currency::getById($currencyId);
		}

		return Currency::getDefault() ?: [];
	}

	/**
	 * Sets module name.
	 *
	 * @param string $name
	 */
	protected function setModuleName(string $name)
	{
		$this->moduleName = $name;
	}
}
